"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResponseActionsClientImpl = exports.HOST_NOT_ENROLLED = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _common = require("@kbn/cases-plugin/common");
var _i18n = require("@kbn/i18n");
var _events = require("../../../../../lib/telemetry/event_based/events");
var _errors = require("../../../../errors");
var _fetch_action_request_by_id = require("../../utils/fetch_action_request_by_id");
var _simple_mem_cache = require("../../../../lib/simple_mem_cache");
var _fetch_action_responses = require("../../utils/fetch_action_responses");
var _create_es_search_iterable = require("../../../../utils/create_es_search_iterable");
var _utils = require("../../utils");
var _is_response_action_supported = require("../../../../../../common/endpoint/service/response_actions/is_response_action_supported");
var _common2 = require("../../../../../../common");
var _action_details_by_id = require("../../action_details_by_id");
var _errors2 = require("../errors");
var _constants = require("../../../../../../common/endpoint/constants");
var _stringify = require("../../../../utils/stringify");
var _constants2 = require("../../../../../../common/constants");
var _translations = require("../../../../utils/translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ELASTIC_RESPONSE_ACTION_MESSAGE = (username = 'system', command, responseActionId) => {
  return _i18n.i18n.translate('xpack.securitySolution.responseActions.comment.message', {
    values: {
      username,
      command,
      responseActionId
    },
    defaultMessage: `Action triggered from Elastic Security by user [{username}] for action [{command} (action id: {responseActionId})]`
  });
};
const ENTERPRISE_LICENSE_REQUIRED_MSG = _i18n.i18n.translate('xpack.securitySolution.responseActionsList.error.licenseTooLow', {
  defaultMessage: 'At least Enterprise license is required to use Response Actions.'
});
const HOST_NOT_ENROLLED = exports.HOST_NOT_ENROLLED = _i18n.i18n.translate('xpack.securitySolution.responseActionsList.error.hostNotEnrolled', {
  defaultMessage: 'The host does not have Elastic Defend integration installed'
});
/**
 * Base class for a Response Actions client
 */
class ResponseActionsClientImpl {
  constructor(options) {
    var _this$constructor$nam;
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "cache", new _simple_mem_cache.SimpleMemCache());
    this.options = options;
    this.log = options.endpointService.createLogger((_this$constructor$nam = this.constructor.name) !== null && _this$constructor$nam !== void 0 ? _this$constructor$nam : 'ResponseActionsClient');
  }

  /**
   * Update cases with information about the hosts that received a response action.
   *
   * **NOTE:** Failures during update will not cause this operation to fail - it will only log the errors
   * @protected
   */
  async updateCases({
    command,
    hosts,
    caseIds = [],
    alertIds = [],
    comment = '',
    actionId
  }) {
    if (caseIds.length === 0 && alertIds.length === 0) {
      this.log.debug(`No updates to Cases needed. 'caseIds' and 'alertIds' are empty`);
      return;
    }
    if (hosts.length === 0) {
      this.log.debug(`No updates to Cases needed. 'hosts' is empty`);
      return;
    }
    const casesClient = this.options.casesClient;
    if (!casesClient) {
      this.log.debug(`No CasesClient available. Skipping updates to Cases!`);
      return;
    }
    const casesFromAlertIds = await Promise.all(alertIds.map(alertID => {
      return casesClient.cases.getCasesByAlertID({
        alertID,
        options: {
          owner: _common2.APP_ID
        }
      }).then(casesFound => {
        return casesFound.map(caseInfo => caseInfo.id);
      }).catch(err => {
        this.log.warn(`Attempt to get cases for alertID [${alertID}][owner: ${_common2.APP_ID}] failed with: ${err.message}`);

        // We don't fail everything here. Just log it and keep going
        return [];
      });
    })).then(results => {
      return results.flat();
    });
    const allCases = [...new Set([...caseIds, ...casesFromAlertIds])];
    if (allCases.length === 0) {
      this.log.debug(`No updates to Cases needed. Alert IDs are not tied to Cases`);
      return;
    }
    this.log.debug(() => `Updating cases:\n${(0, _stringify.stringify)(allCases)}`);
    const attachments = [{
      type: _common.AttachmentType.externalReference,
      externalReferenceId: actionId,
      externalReferenceStorage: {
        type: _common.ExternalReferenceStorageType.elasticSearchDoc
      },
      externalReferenceAttachmentTypeId: _constants2.CASE_ATTACHMENT_ENDPOINT_TYPE_ID,
      externalReferenceMetadata: {
        targets: hosts.map(({
          hostId: endpointId,
          hostname
        }) => {
          return {
            endpointId,
            hostname,
            agentType: this.agentType
          };
        }),
        command,
        comment: comment || _translations.EMPTY_COMMENT
      },
      owner: _common2.APP_ID
    }];
    const casesUpdateResponse = await Promise.all(allCases.map(async caseId => {
      try {
        return await casesClient.attachments.bulkCreate({
          caseId,
          attachments
        });
      } catch (err) {
        this.log.warn(`Attempt to update case ID [${caseId}] failed: ${err.message}\n${(0, _stringify.stringify)(err)}`);
        return null;
      }
    }));
    this.log.debug(() => `Update to cases done:\n${(0, _stringify.stringify)(casesUpdateResponse)}`);
  }
  getMethodOptions(options = {}) {
    return {
      hosts: undefined,
      ruleId: undefined,
      ruleName: undefined,
      error: undefined,
      ...options
    };
  }

  /**
   * Returns the action details for a given response action id
   * @param actionId
   * @protected
   */
  async fetchActionDetails(actionId) {
    return (0, _action_details_by_id.getActionDetailsById)(this.options.esClient, this.options.endpointService.getEndpointMetadataService(), actionId);
  }

  /**
   * Fetches the Action request ES document for a given action id
   * @param actionId
   * @protected
   */
  async fetchActionRequestEsDoc(actionId) {
    const cacheKey = `fetchActionRequestEsDoc-${actionId}`;
    const cachedResponse = this.cache.get(cacheKey);
    if (cachedResponse) {
      this.log.debug(`fetchActionRequestEsDoc(): returning cached response for action id ${actionId}`);
      return cachedResponse;
    }
    return (0, _fetch_action_request_by_id.fetchActionRequestById)(this.options.esClient, actionId).then(actionRequestDoc => {
      this.cache.set(cacheKey, actionRequestDoc);
      return actionRequestDoc;
    });
  }

  /**
   * Fetches the Response Action ES response documents for a given action id
   * @param actionId
   * @param agentIds
   * @protected
   */
  async fetchActionResponseEsDocs(actionId, /** Specific Agent IDs to retrieve. default is to retrieve all */
  agentIds) {
    const responseDocs = await (0, _fetch_action_responses.fetchEndpointActionResponses)({
      esClient: this.options.esClient,
      actionIds: [actionId],
      agentIds
    });
    return responseDocs.reduce((acc, response) => {
      const agentId = Array.isArray(response.agent.id) ? response.agent.id[0] : response.agent.id;
      acc[agentId] = response;
      return acc;
    }, {});
  }

  /**
   * Provides validations against a response action request and returns the result.
   * Checks made should be generic to all response actions and not specific to any one action.
   *
   * @param actionRequest
   * @protected
   */
  async validateRequest(actionRequest) {
    // Validation for Automated Response actions
    if (this.options.isAutomated) {
      // Automated response actions is an Enterprise level feature
      if (!this.options.endpointService.getLicenseService().isEnterprise()) {
        return {
          isValid: false,
          error: new _errors2.ResponseActionsClientError(ENTERPRISE_LICENSE_REQUIRED_MSG, 403)
        };
      }
    }
    if (actionRequest.endpoint_ids.length === 0) {
      return {
        isValid: false,
        error: new _errors2.ResponseActionsClientError(HOST_NOT_ENROLLED, 400)
      };
    }
    if (!(0, _is_response_action_supported.isActionSupportedByAgentType)(this.agentType, actionRequest.command, this.options.isAutomated ? 'automated' : 'manual')) {
      return {
        isValid: false,
        error: new _errors2.ResponseActionsNotSupportedError(actionRequest.command)
      };
    }
    return {
      isValid: true,
      error: undefined
    };
  }

  /**
   * Creates a Response Action request document in the Endpoint index (`.logs-endpoint.actions-default`)
   * @protected
   */
  async writeActionRequestToEndpointIndex(actionRequest) {
    var _actionRequest$error, _actionRequest$commen;
    let errorMsg = String((_actionRequest$error = actionRequest.error) !== null && _actionRequest$error !== void 0 ? _actionRequest$error : '').trim();
    if (!errorMsg) {
      const validation = await this.validateRequest(actionRequest);
      if (!validation.isValid) {
        if (this.options.isAutomated) {
          errorMsg = validation.error.message;
        } else {
          throw validation.error;
        }
      }
    }
    this.notifyUsage(actionRequest.command);
    const doc = {
      '@timestamp': new Date().toISOString(),
      agent: {
        id: actionRequest.endpoint_ids
      },
      EndpointActions: {
        action_id: actionRequest.actionId || (0, _uuid.v4)(),
        expiration: (0, _utils.getActionRequestExpiration)(),
        type: 'INPUT_ACTION',
        input_type: this.agentType,
        data: {
          command: actionRequest.command,
          comment: (_actionRequest$commen = actionRequest.comment) !== null && _actionRequest$commen !== void 0 ? _actionRequest$commen : undefined,
          ...(actionRequest.alert_ids ? {
            alert_id: actionRequest.alert_ids
          } : {}),
          ...(actionRequest.hosts ? {
            hosts: actionRequest.hosts
          } : {}),
          parameters: actionRequest.parameters
        }
      },
      user: {
        id: this.options.username
      },
      meta: actionRequest.meta,
      ...(errorMsg ? {
        error: {
          message: errorMsg
        }
      } : {}),
      ...(actionRequest.ruleId && actionRequest.ruleName ? {
        rule: {
          id: actionRequest.ruleId,
          name: actionRequest.ruleName
        }
      } : {})
    };
    try {
      const logsEndpointActionsResult = await this.options.esClient.index({
        index: _constants.ENDPOINT_ACTIONS_INDEX,
        document: doc,
        refresh: 'wait_for'
      }, {
        meta: true
      });
      if (logsEndpointActionsResult.statusCode !== 201) {
        throw new _errors2.ResponseActionsClientError(`Failed to create (index) action request document. StatusCode: [${logsEndpointActionsResult.statusCode}] Result: ${logsEndpointActionsResult.body.result}`, 500, logsEndpointActionsResult);
      }
      this.sendActionCreationTelemetry(doc);
      return doc;
    } catch (err) {
      this.sendActionCreationErrorTelemetry(actionRequest.command, err);
      if (!(err instanceof _errors2.ResponseActionsClientError)) {
        throw new _errors2.ResponseActionsClientError(`Failed to create action request document: ${err.message}`, 500, err);
      }
      throw err;
    }
  }
  buildActionResponseEsDoc({
    actionId,
    error,
    agentId,
    data,
    meta
  }) {
    const timestamp = new Date().toISOString();
    const doc = {
      '@timestamp': timestamp,
      agent: {
        id: agentId
      },
      EndpointActions: {
        action_id: actionId,
        input_type: this.agentType,
        started_at: timestamp,
        completed_at: timestamp,
        data
      },
      error,
      meta
    };
    return doc;
  }

  /**
   * Writes a Response Action response document to the Endpoint index
   * @param options
   * @protected
   */
  async writeActionResponseToEndpointIndex(options) {
    // FIXME:PT need to ensure we use a index below that has the proper `namespace` when agent type is Endpoint
    //        Background: Endpoint responses require that the document be written to an index that has the
    //        correct `namespace` as defined by the Integration/Agent policy and that logic is not currently implemented.

    const doc = this.buildActionResponseEsDoc(options);
    this.log.debug(() => `Writing response action response:\n${(0, _stringify.stringify)(doc)}`);
    await this.options.esClient.index({
      index: _constants.ENDPOINT_ACTION_RESPONSES_INDEX,
      document: doc,
      refresh: 'wait_for'
    }).catch(err => {
      throw new _errors2.ResponseActionsClientError(`Failed to create action response document: ${err.message}`, 500, err);
    });
    return doc;
  }
  notifyUsage(responseAction) {
    const usageService = this.options.endpointService.getFeatureUsageService();
    const featureKey = usageService.getResponseActionFeatureKey(responseAction);
    if (!featureKey) {
      this.log.warn(`Response action [${responseAction}] does not have a usage feature key defined!`);
      return;
    }
    usageService.notifyUsage(featureKey);
  }

  /**
   * Builds a comment for use in response action requests sent to external EDR systems
   * @protected
   */
  buildExternalComment(actionRequestIndexOptions) {
    const {
      actionId = (0, _uuid.v4)(),
      comment,
      command
    } = actionRequestIndexOptions;

    // If the action request index options does not yet have an actionId assigned to it, then do it now.
    // Need to ensure we have an action id for cross-reference.
    if (!actionRequestIndexOptions.actionId) {
      actionRequestIndexOptions.actionId = actionId;
    }
    return ELASTIC_RESPONSE_ACTION_MESSAGE(this.options.username, command, actionId) + (comment ? `: ${comment}` : '');
  }
  async ensureValidActionId(actionId) {
    const actionRequest = await this.fetchActionRequestEsDoc(actionId);
    if (actionRequest.EndpointActions.input_type !== this.agentType) {
      throw new _errors.NotFoundError(`Action id [${actionId}] with agent type of [${this.agentType}] not found`);
    }
  }
  fetchAllPendingActions() {
    const esClient = this.options.esClient;
    const query = {
      bool: {
        must: {
          // Only actions for this agent type
          term: {
            'EndpointActions.input_type': this.agentType
          }
        },
        must_not: {
          // No action requests that have an `error` property defined
          exists: {
            field: 'error'
          }
        },
        filter: [
        // We only want actions requests whose expiration date is greater than now
        {
          range: {
            'EndpointActions.expiration': {
              gte: 'now'
            }
          }
        }]
      }
    };
    return (0, _create_es_search_iterable.createEsSearchIterable)({
      esClient,
      searchRequest: {
        index: _constants.ENDPOINT_ACTIONS_INDEX,
        sort: '@timestamp',
        query
      },
      resultsMapper: async data => {
        const actionRequests = data.hits.hits.map(hit => hit._source);
        const pendingRequests = [];
        if (actionRequests.length > 0) {
          const actionResults = await (0, _fetch_action_responses.fetchActionResponses)({
            esClient,
            actionIds: actionRequests.map(action => action.EndpointActions.action_id)
          });
          const responsesByActionId = (0, _utils.mapResponsesByActionId)(actionResults);

          // Determine what actions are still pending
          for (const actionRequest of actionRequests) {
            var _responsesByActionId$;
            const actionCompleteInfo = (0, _utils.getActionCompletionInfo)((0, _utils.mapToNormalizedActionRequest)(actionRequest), (_responsesByActionId$ = responsesByActionId[actionRequest.EndpointActions.action_id]) !== null && _responsesByActionId$ !== void 0 ? _responsesByActionId$ : {
              endpointResponses: [],
              fleetResponses: []
            });

            // If not completed, add action to the pending list and calculate the list of agent IDs
            // whose response we are still waiting on
            if (!actionCompleteInfo.isCompleted) {
              const pendingActionData = {
                action: actionRequest,
                pendingAgentIds: []
              };
              for (const [agentId, agentIdState] of Object.entries(actionCompleteInfo.agentState)) {
                if (!agentIdState.isCompleted) {
                  pendingActionData.pendingAgentIds.push(agentId);
                }
              }
              pendingRequests.push(pendingActionData);
            }
          }
        }
        return pendingRequests;
      }
    });
  }
  sendActionCreationTelemetry(actionRequest) {
    var _this$options$isAutom;
    if (!this.options.endpointService.experimentalFeatures.responseActionsTelemetryEnabled) {
      return;
    }
    this.options.endpointService.getTelemetryService().reportEvent(_events.ENDPOINT_RESPONSE_ACTION_SENT_EVENT.eventType, {
      responseActions: {
        actionId: actionRequest.EndpointActions.action_id,
        agentType: this.agentType,
        command: actionRequest.EndpointActions.data.command,
        isAutomated: (_this$options$isAutom = this.options.isAutomated) !== null && _this$options$isAutom !== void 0 ? _this$options$isAutom : false
      }
    });
  }
  sendActionCreationErrorTelemetry(command, error) {
    if (!this.options.endpointService.experimentalFeatures.responseActionsTelemetryEnabled) {
      return;
    }
    this.options.endpointService.getTelemetryService().reportEvent(_events.ENDPOINT_RESPONSE_ACTION_SENT_ERROR_EVENT.eventType, {
      responseActions: {
        agentType: this.agentType,
        command,
        error: error.message
      }
    });
  }
  sendActionResponseTelemetry(responseList) {
    if (!this.options.endpointService.experimentalFeatures.responseActionsTelemetryEnabled) {
      return;
    }
    for (const response of responseList) {
      this.options.endpointService.getTelemetryService().reportEvent(_events.ENDPOINT_RESPONSE_ACTION_STATUS_CHANGE_EVENT.eventType, {
        responseActions: {
          actionId: response.EndpointActions.action_id,
          agentType: this.agentType,
          actionStatus: response.error ? 'failed' : 'successful',
          command: response.EndpointActions.data.command
        }
      });
    }
  }
  async isolate(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('isolate');
  }
  async release(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('unisolate');
  }
  async killProcess(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('kill-process');
  }
  async suspendProcess(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('suspend-process');
  }
  async runningProcesses(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('running-processes');
  }
  async getFile(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('get-file');
  }
  async execute(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('execute');
  }
  async upload(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('upload');
  }
  async scan(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('scan');
  }
  async runscript(actionRequest, options) {
    throw new _errors2.ResponseActionsNotSupportedError('runscript');
  }
  async getCustomScripts() {
    throw new _errors2.ResponseActionsNotSupportedError('getCustomScripts');
  }
  async processPendingActions(_) {
    this.log.debug(`#processPendingActions() method is not implemented for ${this.agentType}!`);
  }
  async getFileDownload(actionId, fileId) {
    throw new _errors2.ResponseActionsNotSupportedError('getFileDownload');
  }
  async getFileInfo(actionId, fileId) {
    throw new _errors2.ResponseActionsNotSupportedError('getFileInfo');
  }
}
exports.ResponseActionsClientImpl = ResponseActionsClientImpl;