"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineGetAllRolesBySpaceRoutes = defineGetAllRolesBySpaceRoutes;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../../common/constants");
var _authorization = require("../../../authorization");
var _errors = require("../../../errors");
var _licensed_route_handler = require("../../licensed_route_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function defineGetAllRolesBySpaceRoutes({
  router,
  authz,
  getFeatures,
  logger,
  buildFlavor,
  subFeaturePrivilegeIterator
}) {
  router.get({
    path: '/internal/security/roles/{spaceId}',
    security: {
      authz: {
        requiredPrivileges: ['manage_spaces']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        spaceId: _configSchema.schema.string({
          minLength: 1
        })
      })
    }
  }, (0, _licensed_route_handler.createLicensedRouteHandler)(async (context, request, response) => {
    try {
      var _ref;
      const hideReservedRoles = buildFlavor === 'serverless';
      const esClient = (await context.core).elasticsearch.client;
      const [features, queryRolesResponse] = await Promise.all([getFeatures(), await esClient.asCurrentUser.security.queryRole({
        query: {
          bool: {
            should: [{
              term: {
                'applications.resources': `space:${request.params.spaceId}`
              }
            }, {
              term: {
                'metadata._reserved': true
              }
            }, {
              bool: {
                must_not: {
                  exists: {
                    field: 'metadata._reserved'
                  }
                }
              }
            }],
            minimum_should_match: 1
          }
        },
        from: 0,
        size: 1000
      })]);
      const elasticsearchRoles = (_ref = queryRolesResponse.roles || []) === null || _ref === void 0 ? void 0 : _ref.reduce((acc, role) => {
        return {
          ...acc,
          [role.name]: role
        };
      }, {});

      // Transform elasticsearch roles into Kibana roles and return in a list sorted by the role name.
      return response.ok({
        body: Object.entries(elasticsearchRoles).reduce((acc, [roleName, elasticsearchRole]) => {
          var _elasticsearchRole$me;
          if (hideReservedRoles && (_elasticsearchRole$me = elasticsearchRole.metadata) !== null && _elasticsearchRole$me !== void 0 && _elasticsearchRole$me._reserved) {
            return acc;
          }
          const role = (0, _authorization.transformElasticsearchRoleToRole)({
            features,
            // @ts-expect-error `remote_cluster` is not known in `Role` type
            elasticsearchRole,
            name: roleName,
            application: authz.applicationName,
            logger,
            subFeaturePrivilegeIterator,
            // For the internal APIs we always transform deprecated privileges.
            replaceDeprecatedKibanaPrivileges: true
          });
          const includeRoleForSpace = role.kibana.some(privilege => {
            const privilegeInSpace = privilege.spaces.includes(request.params.spaceId) || privilege.spaces.includes(_constants.ALL_SPACES_ID);
            if (privilegeInSpace && privilege.base.length) {
              return true;
            }
            const hasFeaturePrivilege = Object.values(privilege.feature).some(featureList => featureList.length);
            return privilegeInSpace && hasFeaturePrivilege;
          });
          if (includeRoleForSpace) {
            acc.push(role);
          }
          return acc;
        }, []).sort(_authorization.compareRolesByName)
      });
    } catch (error) {
      return response.customError((0, _errors.wrapIntoCustomErrorResponse)(error));
    }
  }));
}