"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRoutes = defineRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _api_routes = require("../common/api_routes");
var _pagination = require("../common/pagination");
var _delete_query_rules_ruleset = require("./lib/delete_query_rules_ruleset");
var _delete_query_rules_ruleset_rule = require("./lib/delete_query_rules_ruleset_rule");
var _fetch_indices = require("./lib/fetch_indices");
var _fetch_query_rules_query_rule = require("./lib/fetch_query_rules_query_rule");
var _fetch_query_rules_ruleset = require("./lib/fetch_query_rules_ruleset");
var _fetch_query_rules_sets = require("./lib/fetch_query_rules_sets");
var _is_query_ruleset_exist = require("./lib/is_query_ruleset_exist");
var _put_query_rules_ruleset_set = require("./lib/put_query_rules_ruleset_set");
var _error_handler = require("./utils/error_handler");
var _privilege_check = require("./utils/privilege_check");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function defineRoutes({
  logger,
  router
}) {
  router.get({
    path: _api_routes.APIRoutes.QUERY_RULES_SETS,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: _pagination.DEFAULT_PAGE_VALUE.from
        }),
        size: _configSchema.schema.number({
          defaultValue: _pagination.DEFAULT_PAGE_VALUE.size
        })
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const result = await (0, _fetch_query_rules_sets.fetchQueryRulesSets)(asCurrentUser, {
      from: request.query.from,
      size: request.query.size
    });
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.get({
    path: _api_routes.APIRoutes.QUERY_RULES_RULESET_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        ruleset_id: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const rulesetData = await (0, _fetch_query_rules_ruleset.fetchQueryRulesRuleset)(asCurrentUser, request.params.ruleset_id);
    if (!rulesetData) {
      return response.notFound({
        body: _i18n.i18n.translate('xpack.search.rules.api.routes.rulesetNotFoundErrorMessage', {
          defaultMessage: 'Ruleset not found'
        })
      });
    }
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: rulesetData
    });
  }));
  router.put({
    path: _api_routes.APIRoutes.QUERY_RULES_RULESET_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        ruleset_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        forceWrite: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      body: _configSchema.schema.maybe(_configSchema.schema.object({
        rules: _configSchema.schema.arrayOf(_configSchema.schema.object({
          rule_id: _configSchema.schema.string(),
          type: _configSchema.schema.string(),
          criteria: _configSchema.schema.arrayOf(_configSchema.schema.object({
            type: _configSchema.schema.string(),
            metadata: _configSchema.schema.maybe(_configSchema.schema.string()),
            values: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
          })),
          actions: _configSchema.schema.object({
            ids: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
            docs: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
              _id: _configSchema.schema.string(),
              _index: _configSchema.schema.string()
            })))
          })
        }))
      }))
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    var _request$body;
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const rulesetId = request.params.ruleset_id;
    const forceWrite = request.query.forceWrite;
    const rules = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.rules;
    const isExisting = await (0, _is_query_ruleset_exist.isQueryRulesetExist)(asCurrentUser, rulesetId);
    if (isExisting && !forceWrite) {
      return response.customError({
        statusCode: 409,
        body: _i18n.i18n.translate('xpack.search.rules.api.routes.rulesetAlreadyExistsErrorMessage', {
          defaultMessage: `Ruleset {rulesetId} already exists. Use forceWrite=true to overwrite it.`,
          values: {
            rulesetId
          }
        })
      });
    }
    const result = await (0, _put_query_rules_ruleset_set.putRuleset)(asCurrentUser, rulesetId, rules);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.get({
    path: _api_routes.APIRoutes.QUERY_RULES_RULESET_EXISTS,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        rulesetId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      rulesetId
    } = request.params;
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const isExisting = await (0, _is_query_ruleset_exist.isQueryRulesetExist)(asCurrentUser, rulesetId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: {
        exists: isExisting
      }
    });
  }));
  router.delete({
    path: _api_routes.APIRoutes.QUERY_RULES_RULESET_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        ruleset_id: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const rulesetId = request.params.ruleset_id;
    const result = await (0, _delete_query_rules_ruleset.deleteRuleset)(asCurrentUser, rulesetId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.delete({
    path: _api_routes.APIRoutes.QUERY_RULES_RULESET_RULE,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        ruleset_id: _configSchema.schema.string(),
        rule_id: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const rulesetId = request.params.ruleset_id;
    const ruleId = request.params.rule_id;
    const result = await (0, _delete_query_rules_ruleset_rule.deleteRulesetRule)(asCurrentUser, rulesetId, ruleId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.get({
    path: _api_routes.APIRoutes.QUERY_RULES_QUERY_RULE_FETCH,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        ruleset_id: _configSchema.schema.string(),
        rule_id: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      rule_id: ruleId,
      ruleset_id: rulesetId
    } = request.params;
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const ruleData = await (0, _fetch_query_rules_query_rule.fetchQueryRulesQueryRule)(asCurrentUser, rulesetId, ruleId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: ruleData
    });
  }));
  router.get({
    path: _api_routes.APIRoutes.FETCH_INDICES,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      query: _configSchema.schema.object({
        searchQuery: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      searchQuery
    } = request.query;
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    const {
      indexNames
    } = await (0, _fetch_indices.fetchIndices)(asCurrentUser, searchQuery);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: indexNames
    });
  }));
  router.get({
    path: _api_routes.APIRoutes.FETCH_DOCUMENT,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string(),
        documentId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      indexName,
      documentId
    } = request.params;
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    try {
      const document = await asCurrentUser.get({
        index: indexName,
        id: documentId
      });
      const mappings = await asCurrentUser.indices.getMapping({
        index: indexName
      });
      return response.ok({
        headers: {
          'content-type': 'application/json'
        },
        body: {
          document,
          mappings
        }
      });
    } catch (error) {
      if (error.statusCode === 404) {
        return response.notFound({
          body: `Document with ID ${documentId} not found in index ${indexName}`
        });
      }
      throw error;
    }
  }));
  router.post({
    path: _api_routes.APIRoutes.GENERATE_RULE_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_search_query_rules']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        rulesetId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      rulesetId
    } = request.params;
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    await (0, _privilege_check.checkPrivileges)(core, response);
    for (let i = 0; i < 100; i++) {
      const ruleId = `rule-${Math.floor(Math.random() * 10000).toString().slice(-4)}`;
      // check if it is existing by fetching the rule
      try {
        await asCurrentUser.queryRules.getRule({
          ruleset_id: rulesetId,
          rule_id: ruleId
        });
      } catch (error) {
        // if the rule does not exist return the ruleId
        if (error.statusCode === 404) {
          return response.ok({
            headers: {
              'content-type': 'application/json'
            },
            body: {
              ruleId
            }
          });
        }
        throw error;
      }
    }
    return response.customError({
      statusCode: 409,
      body: _i18n.i18n.translate('xpack.search.rules.api.routes.generateRuleIdErrorMessage', {
        defaultMessage: 'Failed to generate a unique rule ID after 100 attempts.'
      })
    });
  }));
}