"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MlAuditLogger = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ml_client = require("./ml_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const APPLICATION = 'elastic/ml';
const CATEGORY = 'database';
const EVENT_TYPES = {
  creation: 'creation',
  deletion: 'deletion',
  change: 'change',
  access: 'access'
};
class MlAuditLogger {
  constructor(audit, request) {
    (0, _defineProperty2.default)(this, "auditLogger", void 0);
    this.auditLogger = request ? audit.asScoped(request) : audit.withoutRequest;
  }
  async wrapTask(task, taskType, p) {
    try {
      const resp = await task();
      this.logSuccess(taskType, p);
      return resp;
    } catch (error) {
      this.logFailure(taskType, p);
      throw error;
    }
  }
  logMessage(message) {
    this.auditLogger.log({
      message,
      labels: {
        application: APPLICATION
      }
    });
  }
  logSuccess(taskType, p) {
    const entry = this.createLogEntry(taskType, p, true);
    if (entry) {
      this.auditLogger.log(entry);
    }
  }
  logFailure(taskType, p) {
    const entry = this.createLogEntry(taskType, p, false);
    if (entry) {
      this.auditLogger.log(entry);
    }
  }
  createLogEntry(taskType, p, success) {
    try {
      const {
        message,
        type
      } = this.createPartialLogEntry(taskType, p);
      return {
        event: {
          action: taskType,
          type,
          category: [CATEGORY],
          outcome: success ? 'success' : 'failure'
        },
        message,
        labels: {
          application: APPLICATION
        }
      };
    } catch (error) {
      // if an unknown task type is passed, we won't log anything
    }
  }
  createPartialLogEntry(taskType, p) {
    switch (taskType) {
      /* Anomaly Detection */
      case 'ml_put_ad_job':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Creating anomaly detection job ${jobId}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_delete_ad_job':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Deleting anomaly detection job ${jobId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_delete_model_snapshot':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          const [params] = p;
          const snapshotId = params.snapshot_id;
          return {
            message: `Deleting model snapshot ${snapshotId} from job ${jobId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_open_ad_job':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Opening anomaly detection job ${jobId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_close_ad_job':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Closing anomaly detection job ${jobId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_update_ad_job':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Updating anomaly detection job ${jobId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_reset_ad_job':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Resetting anomaly detection job ${jobId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_revert_ad_snapshot':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          const [params] = p;
          const snapshotId = params.snapshot_id;
          return {
            message: `Reverting anomaly detection snapshot ${snapshotId} in job ${jobId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_put_ad_datafeed':
        {
          const [datafeedId] = (0, _ml_client.getDatafeedIdsFromRequest)(p);
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Creating anomaly detection datafeed ${datafeedId} for job ${jobId}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_delete_ad_datafeed':
        {
          const [datafeedId] = (0, _ml_client.getDatafeedIdsFromRequest)(p);
          return {
            message: `Deleting anomaly detection datafeed ${datafeedId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_start_ad_datafeed':
        {
          const [datafeedId] = (0, _ml_client.getDatafeedIdsFromRequest)(p);
          return {
            message: `Starting anomaly detection datafeed ${datafeedId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_stop_ad_datafeed':
        {
          const [datafeedId] = (0, _ml_client.getDatafeedIdsFromRequest)(p);
          return {
            message: `Stopping anomaly detection datafeed ${datafeedId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_update_ad_datafeed':
        {
          const [datafeedId] = (0, _ml_client.getDatafeedIdsFromRequest)(p);
          return {
            message: `Updating anomaly detection datafeed ${datafeedId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_put_calendar':
        {
          var _params$body;
          const [params] = p;
          const calendarId = params.calendar_id;
          // @ts-expect-error body is optional
          const jobIds = ((_params$body = params.body) !== null && _params$body !== void 0 ? _params$body : params).job_ids;
          return {
            message: `Creating calendar ${calendarId} ${jobIds ? `with job(s) ${jobIds}` : ''}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_delete_calendar':
        {
          const [params] = p;
          const calendarId = params.calendar_id;
          return {
            message: `Deleting calendar ${calendarId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_put_calendar_job':
        {
          const [params] = p;
          const calendarId = params.calendar_id;
          const jobIds = params.job_id;
          return {
            message: `Adding job(s) ${jobIds} to calendar ${calendarId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_delete_calendar_job':
        {
          const [params] = p;
          const calendarId = params.calendar_id;
          const jobIds = params.job_id;
          return {
            message: `Removing job(s) ${jobIds} from calendar ${calendarId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_post_calendar_events':
        {
          var _params$body2;
          const [params] = p;
          const calendarId = params.calendar_id;
          // @ts-expect-error body is optional
          const eventsCount = ((_params$body2 = params.body) !== null && _params$body2 !== void 0 ? _params$body2 : params).events;
          return {
            message: `Adding ${eventsCount} event(s) to calendar ${calendarId}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_delete_calendar_event':
        {
          const [params] = p;
          const calendarId = params.calendar_id;
          const eventId = params.event_id;
          return {
            message: `Removing event(s) ${eventId} from calendar ${calendarId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_put_filter':
        {
          const [params] = p;
          const filterId = params.filter_id;
          return {
            message: `Creating filter ${filterId}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_update_filter':
        {
          const [params] = p;
          const filterId = params.filter_id;
          return {
            message: `Updating filter ${filterId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_delete_filter':
        {
          const [params] = p;
          const filterId = params.filter_id;
          return {
            message: `Deleting filter ${filterId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_forecast':
        {
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Forecasting for job ${jobId}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_delete_forecast':
        {
          const [params] = p;
          const forecastId = params.forecast_id;
          const [jobId] = (0, _ml_client.getADJobIdsFromRequest)(p);
          return {
            message: `Deleting forecast ${forecastId} for job ${jobId}`,
            type: [EVENT_TYPES.deletion]
          };
        }

      /* Data Frame Analytics */
      case 'ml_put_dfa_job':
        {
          const [analyticsId] = (0, _ml_client.getDFAJobIdsFromRequest)(p);
          return {
            message: `Creating data frame analytics job ${analyticsId}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_delete_dfa_job':
        {
          const [analyticsId] = (0, _ml_client.getDFAJobIdsFromRequest)(p);
          return {
            message: `Deleting data frame analytics job ${analyticsId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_start_dfa_job':
        {
          const [analyticsId] = (0, _ml_client.getDFAJobIdsFromRequest)(p);
          return {
            message: `Starting data frame analytics job ${analyticsId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_stop_dfa_job':
        {
          const [analyticsId] = (0, _ml_client.getDFAJobIdsFromRequest)(p);
          return {
            message: `Stopping data frame analytics job ${analyticsId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_update_dfa_job':
        {
          const [analyticsId] = (0, _ml_client.getDFAJobIdsFromRequest)(p);
          return {
            message: `Updating data frame analytics job ${analyticsId}`,
            type: [EVENT_TYPES.change]
          };
        }

      /* Trained Models */
      case 'ml_put_trained_model':
        {
          const [modelId] = (0, _ml_client.getModelIdsFromRequest)(p);
          return {
            message: `Creating trained model ${modelId}`,
            type: [EVENT_TYPES.creation]
          };
        }
      case 'ml_delete_trained_model':
        {
          const [modelId] = (0, _ml_client.getModelIdsFromRequest)(p);
          return {
            message: `Deleting trained model ${modelId}`,
            type: [EVENT_TYPES.deletion]
          };
        }
      case 'ml_start_trained_model_deployment':
        {
          const [modelId] = (0, _ml_client.getModelIdsFromRequest)(p);
          return {
            message: `Starting trained model deployment for model ${modelId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_stop_trained_model_deployment':
        {
          const [modelId] = (0, _ml_client.getModelIdsFromRequest)(p);
          return {
            message: `Stopping trained model deployment for model ${modelId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_update_trained_model_deployment':
        {
          const [modelId] = (0, _ml_client.getModelIdsFromRequest)(p);
          return {
            message: `Updating trained model deployment for model ${modelId}`,
            type: [EVENT_TYPES.change]
          };
        }
      case 'ml_infer_trained_model':
        {
          const [modelId] = (0, _ml_client.getModelIdsFromRequest)(p);
          return {
            message: `Inferring trained model ${modelId}`,
            type: [EVENT_TYPES.access]
          };
        }
      default:
        throw new Error(`Unsupported task type: ${taskType}`);
    }
  }
}
exports.MlAuditLogger = MlAuditLogger;