"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withNotifyOnErrors = exports.saveStateInUrlErrorTitle = exports.restoreUrlErrorTitle = exports.flushNotifyOnErrors = void 0;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const restoreUrlErrorTitle = exports.restoreUrlErrorTitle = _i18n.i18n.translate('kibana_utils.stateManagement.url.restoreUrlErrorTitle', {
  defaultMessage: `Error restoring state from URL`
});
const saveStateInUrlErrorTitle = exports.saveStateInUrlErrorTitle = _i18n.i18n.translate('kibana_utils.stateManagement.url.saveStateInUrlErrorTitle', {
  defaultMessage: `Error saving state in URL`
});

// Prevent toast storms by throttling. See https://github.com/elastic/kibana/issues/153073
const throttledOnRestoreError = (0, _lodash.throttle)((toasts, e) => {
  toasts.addError(e, {
    title: restoreUrlErrorTitle
  });
}, 10000);
const throttledOnSaveError = (0, _lodash.throttle)((toasts, e) => {
  toasts.addError(e, {
    title: saveStateInUrlErrorTitle
  });
}, 10000);

// Helper to bypass throttling if consumers need to handle errors right away
const flushNotifyOnErrors = () => {
  throttledOnRestoreError.flush();
  throttledOnSaveError.flush();
};

/**
 * Helper for configuring {@link IKbnUrlStateStorage} to notify about inner errors
 *
 * @example
 * ```ts
 * const kbnUrlStateStorage = createKbnUrlStateStorage({
 *  history,
 *  ...withNotifyOnErrors(core.notifications.toast))
 * }
 * ```
 * @param toast - toastApi from core.notifications.toasts
 */
exports.flushNotifyOnErrors = flushNotifyOnErrors;
const withNotifyOnErrors = toasts => {
  return {
    onGetError: e => throttledOnRestoreError(toasts, e),
    onSetError: e => throttledOnSaveError(toasts, e)
  };
};
exports.withNotifyOnErrors = withNotifyOnErrors;