"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGeneration = exports.ProgressOrder = void 0;
var _fp = require("lodash/fp");
var _react = require("react");
var _lang_smith = require("../../../../../common/lib/lang_smith");
var _generation_error = require("../../../../../../common/api/generation_error");
var _api = require("../../../../../common/lib/api");
var _use_kibana = require("../../../../../common/hooks/use_kibana");
var i18n = _interopRequireWildcard(require("./translations"));
var _telemetry = require("../../../telemetry");
var _constants = require("../../../../../../common/constants");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ProgressOrder = exports.ProgressOrder = ['analyzeLogs', 'ecs', 'categorization', 'related'];
// If the result is classified as a generation error, produce an error message
// as defined in the i18n file. Otherwise, return undefined.
function generationErrorMessage(body) {
  if (!(0, _generation_error.isGenerationErrorBody)(body)) {
    return;
  }
  const errorCode = body.attributes.errorCode;
  if (errorCode === _constants.GenerationErrorCode.CEF_ERROR) {
    if (body.attributes.errorMessageWithLink !== undefined) {
      return {
        link: body.attributes.errorMessageWithLink.link,
        errorMessage: i18n.DECODE_CEF_LINK,
        linkText: body.attributes.errorMessageWithLink.linkText
      };
    }
  }
  const translation = i18n.GENERATION_ERROR_TRANSLATION[errorCode];
  return typeof translation === 'function' ? translation(body.attributes) : translation;
}
const useGeneration = ({
  integrationSettings,
  connector,
  onComplete
}) => {
  const {
    reportGenerationComplete
  } = (0, _telemetry.useTelemetry)();
  const {
    http,
    notifications
  } = (0, _use_kibana.useKibana)().services;
  const [progress, setProgress] = (0, _react.useState)();
  const [error, setError] = (0, _react.useState)(null);
  const [isRequesting, setIsRequesting] = (0, _react.useState)(true);
  (0, _react.useEffect)(() => {
    if (!isRequesting || http == null || connector == null || integrationSettings == null || (notifications === null || notifications === void 0 ? void 0 : notifications.toasts) == null) {
      return;
    }
    const generationStartedAt = Date.now();
    const abortController = new AbortController();
    const deps = {
      http,
      abortSignal: abortController.signal
    };
    (async () => {
      try {
        const result = await runGeneration({
          integrationSettings,
          connector,
          deps,
          setProgress
        });
        const durationMs = Date.now() - generationStartedAt;
        reportGenerationComplete({
          connector,
          integrationSettings,
          durationMs
        });
        onComplete(result);
      } catch (e) {
        var _generationErrorMessa;
        if (abortController.signal.aborted) return;
        const originalErrorMessage = `${e.message}${e.body ? ` (${e.body.statusCode}): ${e.body.message}` : ''}`;
        reportGenerationComplete({
          connector,
          integrationSettings,
          durationMs: Date.now() - generationStartedAt,
          error: originalErrorMessage
        });
        setError((_generationErrorMessa = generationErrorMessage(e.body)) !== null && _generationErrorMessa !== void 0 ? _generationErrorMessa : originalErrorMessage);
      } finally {
        setIsRequesting(false);
      }
    })();
    return () => {
      abortController.abort();
    };
  }, [isRequesting, onComplete, setProgress, connector, http, integrationSettings, reportGenerationComplete, notifications === null || notifications === void 0 ? void 0 : notifications.toasts]);
  const retry = (0, _react.useCallback)(() => {
    setError(null);
    setIsRequesting(true);
  }, []);
  return {
    progress,
    error,
    retry
  };
};
exports.useGeneration = useGeneration;
async function runGeneration({
  integrationSettings,
  connector,
  deps,
  setProgress
}) {
  var _integrationSettings$6, _integrationSettings$7, _logSamples, _samplesFormat, _additionalProcessors, _samplesFormat2;
  let additionalProcessors;
  // logSamples may be modified to JSON format if they are in different formats
  // Keeping originalLogSamples for running pipeline and generating docs
  const originalLogSamples = integrationSettings.logSamples;
  let logSamples = integrationSettings.logSamples;
  let samplesFormat = integrationSettings.samplesFormat;
  if (integrationSettings.samplesFormat === undefined) {
    var _integrationSettings$, _integrationSettings$2, _ref, _integrationSettings$3, _ref2, _integrationSettings$4, _integrationSettings$5;
    const analyzeLogsRequest = {
      packageName: (_integrationSettings$ = integrationSettings.name) !== null && _integrationSettings$ !== void 0 ? _integrationSettings$ : '',
      dataStreamName: (_integrationSettings$2 = integrationSettings.dataStreamName) !== null && _integrationSettings$2 !== void 0 ? _integrationSettings$2 : '',
      packageTitle: (_ref = (_integrationSettings$3 = integrationSettings.title) !== null && _integrationSettings$3 !== void 0 ? _integrationSettings$3 : integrationSettings.name) !== null && _ref !== void 0 ? _ref : '',
      dataStreamTitle: (_ref2 = (_integrationSettings$4 = integrationSettings.dataStreamTitle) !== null && _integrationSettings$4 !== void 0 ? _integrationSettings$4 : integrationSettings.dataStreamName) !== null && _ref2 !== void 0 ? _ref2 : '',
      logSamples: (_integrationSettings$5 = integrationSettings.logSamples) !== null && _integrationSettings$5 !== void 0 ? _integrationSettings$5 : [],
      connectorId: connector.id,
      langSmithOptions: (0, _lang_smith.getLangSmithOptions)()
    };
    setProgress('analyzeLogs');
    const analyzeLogsResult = await (0, _api.runAnalyzeLogsGraph)(analyzeLogsRequest, deps);
    if ((0, _fp.isEmpty)(analyzeLogsResult === null || analyzeLogsResult === void 0 ? void 0 : analyzeLogsResult.results)) {
      throw new Error('No results from Analyze Logs Graph');
    }
    logSamples = analyzeLogsResult.results.parsedSamples;
    samplesFormat = analyzeLogsResult.results.samplesFormat;
    additionalProcessors = analyzeLogsResult.additionalProcessors;
  }
  const ecsRequest = {
    packageName: (_integrationSettings$6 = integrationSettings.name) !== null && _integrationSettings$6 !== void 0 ? _integrationSettings$6 : '',
    dataStreamName: (_integrationSettings$7 = integrationSettings.dataStreamName) !== null && _integrationSettings$7 !== void 0 ? _integrationSettings$7 : '',
    rawSamples: (_logSamples = logSamples) !== null && _logSamples !== void 0 ? _logSamples : [],
    samplesFormat: (_samplesFormat = samplesFormat) !== null && _samplesFormat !== void 0 ? _samplesFormat : {
      name: 'json'
    },
    additionalProcessors: (_additionalProcessors = additionalProcessors) !== null && _additionalProcessors !== void 0 ? _additionalProcessors : [],
    connectorId: connector.id,
    langSmithOptions: (0, _lang_smith.getLangSmithOptions)()
  };
  setProgress('ecs');
  const ecsGraphResult = await (0, _api.runEcsGraph)(ecsRequest, deps);
  if ((0, _fp.isEmpty)(ecsGraphResult === null || ecsGraphResult === void 0 ? void 0 : ecsGraphResult.results)) {
    throw new Error('No results from ECS graph');
  }
  const categorizationRequest = {
    ...ecsRequest,
    rawSamples: originalLogSamples !== null && originalLogSamples !== void 0 ? originalLogSamples : [],
    samplesFormat: (_samplesFormat2 = samplesFormat) !== null && _samplesFormat2 !== void 0 ? _samplesFormat2 : {
      name: 'json'
    },
    currentPipeline: ecsGraphResult.results.pipeline
  };
  setProgress('categorization');
  const categorizationResult = await (0, _api.runCategorizationGraph)(categorizationRequest, deps);
  const relatedRequest = {
    ...categorizationRequest,
    currentPipeline: categorizationResult.results.pipeline
  };
  setProgress('related');
  const relatedGraphResult = await (0, _api.runRelatedGraph)(relatedRequest, deps);
  if ((0, _fp.isEmpty)(relatedGraphResult === null || relatedGraphResult === void 0 ? void 0 : relatedGraphResult.results)) {
    throw new Error('Results not found in response');
  }
  return {
    pipeline: relatedGraphResult.results.pipeline,
    docs: relatedGraphResult.results.docs,
    samplesFormat
  };
}