"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FeatureFlagsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _std = require("@kbn/std");
var _serverSdk = require("@openfeature/server-sdk");
var _deepmerge = _interopRequireDefault(require("deepmerge"));
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _create_open_feature_logger = require("./create_open_feature_logger");
var _set_provider_with_retries = require("./set_provider_with_retries");
var _feature_flags_config = require("./feature_flags_config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Core-internal contract for the setup lifecycle step.
 * @internal
 */

/**
 * The server-side Feature Flags Service
 * @internal
 */
class FeatureFlagsService {
  /**
   * The core service's constructor
   * @param core {@link CoreContext}
   */
  constructor(core) {
    (0, _defineProperty2.default)(this, "featureFlagsClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "overrides$", new _rxjs.BehaviorSubject({}));
    (0, _defineProperty2.default)(this, "context", {
      kind: 'multi'
    });
    (0, _defineProperty2.default)(this, "initialFeatureFlagsGetter", async () => ({}));
    this.core = core;
    this.logger = core.logger.get('feature-flags-service');
    this.featureFlagsClient = _serverSdk.OpenFeature.getClient();
    _serverSdk.OpenFeature.setLogger((0, _create_open_feature_logger.createOpenFeatureLogger)(this.logger.get('open-feature')));
  }

  /**
   * Setup lifecycle method
   */
  setup() {
    // Register "overrides" to be changed via the dynamic config endpoint (enabled in test environments only)
    this.core.configService.addDynamicConfigPaths(_feature_flags_config.featureFlagsConfig.path, ['overrides']);
    this.core.configService.atPath(_feature_flags_config.featureFlagsConfig.path).subscribe(({
      overrides = {}
    }) => {
      this.overrides$.next((0, _std.getFlattenedObject)(overrides));
    });
    return {
      getOverrides: () => this.overrides$.value,
      getInitialFeatureFlags: () => this.initialFeatureFlagsGetter(),
      setInitialFeatureFlagsGetter: getter => {
        this.initialFeatureFlagsGetter = getter;
      },
      setProvider: provider => {
        if (_serverSdk.OpenFeature.providerMetadata !== _serverSdk.NOOP_PROVIDER.metadata) {
          throw new Error('A provider has already been set. This API cannot be called twice.');
        }
        (0, _set_provider_with_retries.setProviderWithRetries)(provider, this.logger);
      },
      appendContext: contextToAppend => this.appendContext(contextToAppend)
    };
  }

  /**
   * Start lifecycle method
   */
  start() {
    const featureFlagsChanged$ = new _rxjs.Subject();
    this.featureFlagsClient.addHandler(_serverSdk.ServerProviderEvents.ConfigurationChanged, event => {
      if (event !== null && event !== void 0 && event.flagsChanged) {
        featureFlagsChanged$.next(event.flagsChanged);
      }
    });
    this.overrides$.pipe((0, _rxjs.pairwise)()).subscribe(([prev, next]) => {
      const mergedObject = {
        ...prev,
        ...next
      };
      const keys = Object.keys(mergedObject).filter(
      // Keep only the keys that have been removed or changed
      key => !Object.hasOwn(next, key) || next[key] !== prev[key]);
      featureFlagsChanged$.next(keys);
    });
    const observeFeatureFlag$ = flagName => featureFlagsChanged$.pipe((0, _rxjs.filter)(flagNames => flagNames.includes(flagName)), (0, _rxjs.startWith)([flagName]),
    // only to emit on the first call
    (0, _rxjs.takeUntil)(this.stop$) // stop the observable when the service stops
    );
    return {
      appendContext: contextToAppend => this.appendContext(contextToAppend),
      getBooleanValue: async (flagName, fallbackValue) => this.evaluateFlag(this.featureFlagsClient.getBooleanValue, flagName, fallbackValue),
      getStringValue: async (flagName, fallbackValue) => await this.evaluateFlag(this.featureFlagsClient.getStringValue, flagName, fallbackValue),
      getNumberValue: async (flagName, fallbackValue) => await this.evaluateFlag(this.featureFlagsClient.getNumberValue, flagName, fallbackValue),
      getBooleanValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.switchMap)(() => this.evaluateFlag(this.featureFlagsClient.getBooleanValue, flagName, fallbackValue)));
      },
      getStringValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.switchMap)(() => this.evaluateFlag(this.featureFlagsClient.getStringValue, flagName, fallbackValue)));
      },
      getNumberValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.switchMap)(() => this.evaluateFlag(this.featureFlagsClient.getNumberValue, flagName, fallbackValue)));
      }
    };
  }

  /**
   * Stop lifecycle method
   */
  async stop() {
    await _serverSdk.OpenFeature.close();
    this.overrides$.complete();
    this.stop$.next();
    this.stop$.complete();
  }

  /**
   * Wrapper to evaluate flags with the common config overrides interceptions + APM and counters reporting
   * @param evaluationFn The actual evaluation API
   * @param flagName The name of the flag to evaluate
   * @param fallbackValue The fallback value
   * @internal
   */
  async evaluateFlag(evaluationFn, flagName, fallbackValue) {
    const override = (0, _lodash.get)(this.overrides$.value, flagName); // using lodash get because flagName can come with dots and the config parser might structure it in objects.
    const value = typeof override !== 'undefined' ? override :
    // We have to bind the evaluation or the client will lose its internal context
    await evaluationFn.bind(this.featureFlagsClient)(flagName, fallbackValue);
    _elasticApmNode.default.addLabels({
      [`flag_${flagName.replaceAll('.', '_')}`]: value
    });
    // TODO: increment usage counter
    return value;
  }

  /**
   * Formats the provided context to fulfill the expected multi-context structure.
   * @param contextToAppend The {@link EvaluationContext} to append.
   * @internal
   */
  appendContext(contextToAppend) {
    // If no kind provided, default to the project|deployment level.
    const {
      kind = 'kibana',
      ...rest
    } = contextToAppend;
    // Format the context to fulfill the expected multi-context structure
    const formattedContextToAppend = kind === 'multi' ? contextToAppend : {
      kind: 'multi',
      [kind]: rest
    };

    // Merge the formatted context to append to the global context, and set it in the OpenFeature client.
    this.context = (0, _deepmerge.default)(this.context, formattedContextToAppend);
    _serverSdk.OpenFeature.setContext(this.context);
  }
}
exports.FeatureFlagsService = FeatureFlagsService;