"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSyntheticsSuggestionsRoute = void 0;
var _saved_objects = require("../../../common/types/saved_objects");
var _runtime_types = require("../../../common/runtime_types");
var _constants = require("../../../common/constants");
var _common = require("../common");
var _get_all_locations = require("../../synthetics_service/get_all_locations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Helper to sum buckets by key
function sumBuckets(bucketsA = [], bucketsB = []) {
  const map = new Map();
  for (const {
    key,
    doc_count: docCount
  } of bucketsA) {
    map.set(key, docCount);
  }
  for (const {
    key,
    doc_count: docCount
  } of bucketsB) {
    map.set(key, (map.get(key) || 0) + docCount);
  }
  return Array.from(map.entries()).map(([key, docCount]) => ({
    key,
    doc_count: docCount
  }));
}

// Helper to sum monitorIdsAggs buckets
function sumMonitorIdsBuckets(bucketsA = [], bucketsB = []) {
  const map = new Map();
  for (const b of bucketsA) {
    map.set(b.key, {
      doc_count: b.doc_count,
      name: b.name
    });
  }
  for (const b of bucketsB) {
    if (map.has(b.key)) {
      map.get(b.key).doc_count += b.doc_count;
    } else {
      map.set(b.key, {
        doc_count: b.doc_count,
        name: b.name
      });
    }
  }
  return Array.from(map.entries()).map(([key, {
    doc_count: docCount,
    name
  }]) => ({
    key,
    doc_count: docCount,
    name
  }));
}

// Helper to generate aggs for new or legacy monitors
function getAggs(isLegacy) {
  const attributes = isLegacy ? _saved_objects.legacyMonitorAttributes : _saved_objects.syntheticsMonitorAttributes;
  const savedObjectType = isLegacy ? _saved_objects.legacySyntheticsMonitorTypeSingle : _saved_objects.syntheticsMonitorSavedObjectType;
  return {
    tagsAggs: {
      terms: {
        field: `${attributes}.${_runtime_types.ConfigKey.TAGS}`,
        size: 10000,
        exclude: ['']
      }
    },
    monitorTypeAggs: {
      terms: {
        field: `${attributes}.${_runtime_types.ConfigKey.MONITOR_TYPE}.keyword`,
        size: 10000,
        exclude: ['']
      }
    },
    locationsAggs: {
      terms: {
        field: `${attributes}.${_runtime_types.ConfigKey.LOCATIONS}.id`,
        size: 10000,
        exclude: ['']
      }
    },
    projectsAggs: {
      terms: {
        field: `${attributes}.${_runtime_types.ConfigKey.PROJECT_ID}`,
        size: 10000,
        exclude: ['']
      }
    },
    monitorTypesAggs: {
      terms: {
        field: `${attributes}.${_runtime_types.ConfigKey.MONITOR_TYPE}.keyword`,
        size: 10000,
        exclude: ['']
      }
    },
    monitorIdsAggs: {
      terms: {
        field: `${attributes}.${_runtime_types.ConfigKey.MONITOR_QUERY_ID}`,
        size: 10000,
        exclude: ['']
      },
      aggs: {
        name: {
          top_hits: {
            _source: [`${savedObjectType}.${_runtime_types.ConfigKey.NAME}`],
            size: 1
          }
        }
      }
    }
  };
}
const getSyntheticsSuggestionsRoute = () => ({
  method: 'GET',
  path: _constants.SYNTHETICS_API_URLS.SUGGESTIONS,
  validate: {
    query: _common.QuerySchema
  },
  handler: async route => {
    var _ref, _ref2, _summedTags$map, _summedLocations$map, _summedProjects$map, _summedMonitorTypes$m;
    const {
      savedObjectsClient
    } = route;
    const {
      query
    } = route.request.query;
    const {
      filtersStr
    } = await (0, _common.getMonitorFilters)(route, _saved_objects.syntheticsMonitorAttributes);
    const {
      allLocations = []
    } = await (0, _get_all_locations.getAllLocations)(route);

    // Find for new monitors
    const data = await savedObjectsClient.find({
      type: _saved_objects.syntheticsMonitorSavedObjectType,
      perPage: 0,
      filter: filtersStr ? filtersStr : undefined,
      aggs: getAggs(false),
      search: query ? `${query}*` : undefined,
      searchFields: _common.MONITOR_SEARCH_FIELDS
    });
    const {
      filtersStr: legacyFilterStr
    } = await (0, _common.getMonitorFilters)(route, _saved_objects.legacyMonitorAttributes);

    // Find for legacy monitors
    const legacyData = await savedObjectsClient.find({
      type: _saved_objects.legacySyntheticsMonitorTypeSingle,
      perPage: 0,
      filter: legacyFilterStr ? legacyFilterStr : undefined,
      aggs: getAggs(true),
      search: query ? `${query}*` : undefined,
      searchFields: _common.MONITOR_SEARCH_FIELDS
    });

    // Extract aggs
    const {
      monitorTypesAggs,
      tagsAggs,
      locationsAggs,
      projectsAggs,
      monitorIdsAggs
    } = (_ref = data === null || data === void 0 ? void 0 : data.aggregations) !== null && _ref !== void 0 ? _ref : {};
    const {
      monitorTypesAggs: legacyMonitorTypesAggs,
      tagsAggs: legacyTagsAggs,
      locationsAggs: legacyLocationsAggs,
      projectsAggs: legacyProjectsAggs,
      monitorIdsAggs: legacyMonitorIdsAggs
    } = (_ref2 = legacyData === null || legacyData === void 0 ? void 0 : legacyData.aggregations) !== null && _ref2 !== void 0 ? _ref2 : {};
    const allLocationsMap = new Map(allLocations.map(obj => [obj.id, obj.label]));

    // Sum buckets
    const summedTags = sumBuckets(tagsAggs === null || tagsAggs === void 0 ? void 0 : tagsAggs.buckets, legacyTagsAggs === null || legacyTagsAggs === void 0 ? void 0 : legacyTagsAggs.buckets);
    const summedLocations = sumBuckets(locationsAggs === null || locationsAggs === void 0 ? void 0 : locationsAggs.buckets, legacyLocationsAggs === null || legacyLocationsAggs === void 0 ? void 0 : legacyLocationsAggs.buckets);
    const summedProjects = sumBuckets(projectsAggs === null || projectsAggs === void 0 ? void 0 : projectsAggs.buckets, legacyProjectsAggs === null || legacyProjectsAggs === void 0 ? void 0 : legacyProjectsAggs.buckets);
    const summedMonitorTypes = sumBuckets(monitorTypesAggs === null || monitorTypesAggs === void 0 ? void 0 : monitorTypesAggs.buckets, legacyMonitorTypesAggs === null || legacyMonitorTypesAggs === void 0 ? void 0 : legacyMonitorTypesAggs.buckets);
    const summedMonitorIds = sumMonitorIdsBuckets(monitorIdsAggs === null || monitorIdsAggs === void 0 ? void 0 : monitorIdsAggs.buckets, legacyMonitorIdsAggs === null || legacyMonitorIdsAggs === void 0 ? void 0 : legacyMonitorIdsAggs.buckets);
    return {
      monitorIds: summedMonitorIds === null || summedMonitorIds === void 0 ? void 0 : summedMonitorIds.map(({
        key,
        doc_count: count,
        name
      }) => {
        var _name$hits, _name$hits$hits$, _source$syntheticsMon, _source$legacySynthet;
        const source = (name === null || name === void 0 ? void 0 : (_name$hits = name.hits) === null || _name$hits === void 0 ? void 0 : (_name$hits$hits$ = _name$hits.hits[0]) === null || _name$hits$hits$ === void 0 ? void 0 : _name$hits$hits$._source) || {};
        return {
          label: (source === null || source === void 0 ? void 0 : (_source$syntheticsMon = source[_saved_objects.syntheticsMonitorSavedObjectType]) === null || _source$syntheticsMon === void 0 ? void 0 : _source$syntheticsMon[_runtime_types.ConfigKey.NAME]) || (source === null || source === void 0 ? void 0 : (_source$legacySynthet = source[_saved_objects.legacySyntheticsMonitorTypeSingle]) === null || _source$legacySynthet === void 0 ? void 0 : _source$legacySynthet[_runtime_types.ConfigKey.NAME]) || key,
          value: key,
          count
        };
      }),
      tags: (_summedTags$map = summedTags === null || summedTags === void 0 ? void 0 : summedTags.map(({
        key,
        doc_count: count
      }) => ({
        label: key,
        value: key,
        count
      }))) !== null && _summedTags$map !== void 0 ? _summedTags$map : [],
      locations: (_summedLocations$map = summedLocations === null || summedLocations === void 0 ? void 0 : summedLocations.map(({
        key,
        doc_count: count
      }) => ({
        label: allLocationsMap.get(key) || key,
        value: key,
        count
      }))) !== null && _summedLocations$map !== void 0 ? _summedLocations$map : [],
      projects: (_summedProjects$map = summedProjects === null || summedProjects === void 0 ? void 0 : summedProjects.map(({
        key,
        doc_count: count
      }) => ({
        label: key,
        value: key,
        count
      }))) !== null && _summedProjects$map !== void 0 ? _summedProjects$map : [],
      monitorTypes: (_summedMonitorTypes$m = summedMonitorTypes === null || summedMonitorTypes === void 0 ? void 0 : summedMonitorTypes.map(({
        key,
        doc_count: count
      }) => ({
        label: key,
        value: key,
        count
      }))) !== null && _summedMonitorTypes$m !== void 0 ? _summedMonitorTypes$m : []
    };
  }
});
exports.getSyntheticsSuggestionsRoute = getSyntheticsSuggestionsRoute;