"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSyntheticsParamsRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _saved_objects = require("../../../../common/types/saved_objects");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RequestParamsSchema = _configSchema.schema.object({
  id: _configSchema.schema.maybe(_configSchema.schema.string())
});
const getSyntheticsParamsRoute = () => ({
  method: 'GET',
  path: _constants.SYNTHETICS_API_URLS.PARAMS + '/{id?}',
  validate: {},
  validation: {
    request: {
      params: RequestParamsSchema
    }
  },
  handler: async routeContext => {
    const {
      savedObjectsClient,
      request,
      response,
      spaceId
    } = routeContext;
    try {
      const {
        id: paramId
      } = request.params;
      if (await isAnAdminUser(routeContext)) {
        return getDecryptedParams(routeContext, paramId);
      } else {
        if (paramId) {
          const savedObject = await savedObjectsClient.get(_saved_objects.syntheticsParamType, paramId);
          return toClientResponse(savedObject);
        }
        return findAllParams(routeContext);
      }
    } catch (error) {
      var _error$output;
      if (((_error$output = error.output) === null || _error$output === void 0 ? void 0 : _error$output.statusCode) === 404) {
        return response.notFound({
          body: {
            message: `Kibana space '${spaceId}' does not exist`
          }
        });
      }
      throw error;
    }
  }
});
exports.getSyntheticsParamsRoute = getSyntheticsParamsRoute;
const isAnAdminUser = async routeContext => {
  var _await$server$coreSta, _server$coreStart;
  const {
    request,
    server
  } = routeContext;
  const user = server.coreStart.security.authc.getCurrentUser(request);
  const isSuperUser = user === null || user === void 0 ? void 0 : user.roles.includes('superuser');
  const isAdmin = user === null || user === void 0 ? void 0 : user.roles.includes('kibana_admin');
  const canSave = (_await$server$coreSta = (await ((_server$coreStart = server.coreStart) === null || _server$coreStart === void 0 ? void 0 : _server$coreStart.capabilities.resolveCapabilities(request, {
    capabilityPath: 'uptime.*'
  }))).uptime.save) !== null && _await$server$coreSta !== void 0 ? _await$server$coreSta : false;
  return (isSuperUser || isAdmin) && canSave;
};
const getDecryptedParams = async ({
  server,
  spaceId
}, paramId) => {
  const encryptedSavedObjectsClient = server.encryptedSavedObjects.getClient();
  if (paramId) {
    const savedObject = await encryptedSavedObjectsClient.getDecryptedAsInternalUser(_saved_objects.syntheticsParamType, paramId, {
      namespace: spaceId
    });
    return toClientResponse(savedObject);
  }
  const finder = await encryptedSavedObjectsClient.createPointInTimeFinderDecryptedAsInternalUser({
    type: _saved_objects.syntheticsParamType,
    perPage: 1000,
    namespaces: [spaceId]
  });
  const hits = [];
  for await (const result of finder.find()) {
    hits.push(...result.saved_objects.map(toClientResponse));
  }
  void finder.close();
  return hits;
};
const findAllParams = async ({
  savedObjectsClient
}) => {
  const finder = savedObjectsClient.createPointInTimeFinder({
    type: _saved_objects.syntheticsParamType,
    perPage: 1000
  });
  const hits = [];
  for await (const result of finder.find()) {
    hits.push(...result.saved_objects.map(toClientResponse));
  }
  void finder.close();
  return hits;
};
const toClientResponse = savedObject => {
  const {
    id,
    attributes,
    namespaces
  } = savedObject;
  return {
    ...attributes,
    id,
    namespaces
  };
};