"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGetDataStreamStatuses = useGetDataStreamStatuses;
var _public = require("@kbn/observability-shared-plugin/public");
var _react = require("react");
var _lodash = require("lodash");
var _api = require("./api");
var _contexts = require("../../../contexts");
var _use_object_metrics = require("../../step_details_page/hooks/use_object_metrics");
var _common = require("../data_retention/common");
var _policy_labels = require("../data_retention/policy_labels");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Query the Data Streams API for stats and info about the current data streams.
 *
 * Filters out non-synthetics streams and overlays additional useful fields for display.
 */
function useGetDataStreamStatuses() {
  const {
    lastRefresh
  } = (0, _react.useContext)(_contexts.SyntheticsRefreshContext);
  const {
    data,
    error,
    loading
  } = (0, _public.useFetcher)(_api.getDslPolicies, [lastRefresh]);
  if (!Array.isArray(data) || !!error) return {
    dataStreamStatuses: undefined,
    error,
    loading
  };
  const dataStreamMap = (0, _lodash.keyBy)(data, 'indexTemplateName');
  const dataStreamStatuses = [];
  let totalBytes = 0;
  let summaryItem = null;
  for (const {
    indexTemplate,
    label
  } of _policy_labels.policyLabels) {
    const dataStream = dataStreamMap[indexTemplate];
    if (dataStream) {
      var _dataStream$storageSi;
      dataStreamStatuses.push(formatDataStreamInfo(dataStream));
      totalBytes += (_dataStream$storageSi = dataStream.storageSizeBytes) !== null && _dataStream$storageSi !== void 0 ? _dataStream$storageSi : 0;
    } else {
      const missingStream = toMissingDataStream({
        indexTemplate,
        label
      });
      dataStreamStatuses.push(missingStream);
      if (indexTemplate === 'synthetics') summaryItem = missingStream;
    }
  }
  if (summaryItem !== null) summaryItem.storageSize = (0, _use_object_metrics.formatBytes)(totalBytes);
  dataStreamStatuses.sort(({
    indexTemplateName: a
  }, {
    indexTemplateName: b
  }) => a.localeCompare(b));
  return {
    dataStreamStatuses,
    error,
    loading
  };
}

/**
 * The table expects entries even when the data stream is missing, so we need to create a placeholder.
 */
function toMissingDataStream({
  label: name,
  indexTemplate: indexTemplateName
}) {
  return {
    isEmpty: true,
    name,
    indices: [],
    indexTemplateName,
    storageSize: (0, _use_object_metrics.formatBytes)(0),
    storageSizeBytes: 0,
    lifecycle: {
      enabled: false,
      data_retention: indexTemplateName === 'synthetics' ? '--' : ''
    },
    timeStampField: {
      name: '@timestamp'
    },
    generation: 0,
    health: 'green',
    privileges: {
      delete_index: true,
      manage_data_stream_lifecycle: true,
      read_failure_store: true
    },
    hidden: false,
    nextGenerationManagedBy: 'Data stream lifecycle',
    indexMode: 'standard'
  };
}

/**
 * Overlay inferred fields to match the table's expectations.
 */
function formatDataStreamInfo({
  name,
  lifecycle,
  indexTemplateName,
  storageSizeBytes,
  ...rest
}) {
  var _policyLabel$label;
  const policyLabel = _policy_labels.policyLabels.find(({
    indexTemplate
  }) => indexTemplate === indexTemplateName);
  return {
    ...rest,
    name: (_policyLabel$label = policyLabel === null || policyLabel === void 0 ? void 0 : policyLabel.label) !== null && _policyLabel$label !== void 0 ? _policyLabel$label : name,
    dataStreamName: name,
    indexTemplateName,
    storageSize: storageSizeBytes !== undefined ? (0, _use_object_metrics.formatBytes)(storageSizeBytes) : '',
    storageSizeBytes,
    lifecycle: {
      ...lifecycle,
      data_retention: (lifecycle === null || lifecycle === void 0 ? void 0 : lifecycle.data_retention) !== undefined ? (0, _common.formatAge)(String(lifecycle === null || lifecycle === void 0 ? void 0 : lifecycle.data_retention)) : ''
    }
  };
}