"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.recalcColumnWidths = exports.processorConverter = exports.isManualIngestPipelineJsonProcessor = exports.isGrokProcessor = exports.isDissectProcessor = exports.isDateProcessor = exports.getFormStateFrom = exports.getDefaultFormStateByType = exports.dataSourceConverter = exports.convertFormStateToProcessor = exports.SPECIALISED_TYPES = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _eui = require("@elastic/eui");
var _lodash = require("lodash");
var _grokUi = require("@kbn/grok-ui");
var _condition = require("../../../util/condition");
var _config_driven = require("./processors/config_driven");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/naming-convention */

/**
 * These are processor types with specialised UI. Other processor types are handled by a generic config-driven UI.
 */
const SPECIALISED_TYPES = exports.SPECIALISED_TYPES = ['date', 'dissect', 'grok'];
const PRIORITIZED_CONTENT_FIELDS = ['message', 'body.text', 'error.message', 'event.original', 'attributes.exception.message'];
const PRIORITIZED_DATE_FIELDS = ['timestamp', 'logtime', 'initial_date', 'date', 'event.time.received', 'event.ingested'];
const getDefaultTextField = (sampleDocs, prioritizedFields) => {
  // Count occurrences of well-known text fields in the sample documents
  const acceptableDefaultFields = sampleDocs.flatMap(doc => Object.keys(doc).filter(key => prioritizedFields.includes(key)));
  const acceptableFieldsOccurrences = (0, _lodash.countBy)(acceptableDefaultFields);

  // Sort by count descending first, then by order of field in prioritizedFields
  const sortedFields = (0, _lodash.orderBy)(Object.entries(acceptableFieldsOccurrences), [([_field, occurrencies]) => occurrencies,
  // Sort entries by occurrencies descending
  ([field]) => prioritizedFields.indexOf(field) // Sort entries by priority order in well-known fields
  ], ['desc', 'asc']);
  const mostCommonField = sortedFields[0];
  return mostCommonField ? mostCommonField[0] : '';
};
const defaultDateProcessorFormState = sampleDocs => ({
  type: 'date',
  field: getDefaultTextField(sampleDocs, PRIORITIZED_DATE_FIELDS),
  formats: [],
  locale: '',
  target_field: '',
  timezone: '',
  output_format: '',
  ignore_failure: true,
  if: _condition.ALWAYS_CONDITION
});
const defaultDissectProcessorFormState = sampleDocs => ({
  type: 'dissect',
  field: getDefaultTextField(sampleDocs, PRIORITIZED_CONTENT_FIELDS),
  pattern: '',
  ignore_failure: true,
  ignore_missing: true,
  if: _condition.ALWAYS_CONDITION
});
const defaultGrokProcessorFormState = (sampleDocs, formStateDependencies) => ({
  type: 'grok',
  field: getDefaultTextField(sampleDocs, PRIORITIZED_CONTENT_FIELDS),
  patterns: [new _grokUi.DraftGrokExpression(formStateDependencies.grokCollection, '')],
  pattern_definitions: {},
  ignore_failure: true,
  ignore_missing: true,
  if: _condition.ALWAYS_CONDITION
});
const defaultManualIngestPipelineProcessorFormState = () => ({
  type: 'manual_ingest_pipeline',
  processors: [],
  ignore_failure: true,
  if: _condition.ALWAYS_CONDITION
});
const configDrivenDefaultFormStates = (0, _lodash.mapValues)(_config_driven.configDrivenProcessors, config => () => config.defaultFormState);
const defaultProcessorFormStateByType = {
  date: defaultDateProcessorFormState,
  dissect: defaultDissectProcessorFormState,
  grok: defaultGrokProcessorFormState,
  manual_ingest_pipeline: defaultManualIngestPipelineProcessorFormState,
  ...configDrivenDefaultFormStates
};
const getDefaultFormStateByType = (type, sampleDocuments, formStateDependencies) => defaultProcessorFormStateByType[type](sampleDocuments, formStateDependencies);
exports.getDefaultFormStateByType = getDefaultFormStateByType;
const getFormStateFrom = (sampleDocuments, formStateDependencies, processor) => {
  if (!processor) return defaultGrokProcessorFormState(sampleDocuments, formStateDependencies);
  if (isGrokProcessor(processor)) {
    const {
      grok
    } = processor;
    const clone = structuredClone({
      ...(0, _lodash.omit)(grok, 'patterns'),
      patterns: [],
      type: 'grok'
    });
    clone.patterns = grok.patterns.map(pattern => new _grokUi.DraftGrokExpression(formStateDependencies.grokCollection, pattern));
    return clone;
  }
  if (isDissectProcessor(processor)) {
    const {
      dissect
    } = processor;
    return structuredClone({
      ...dissect,
      type: 'dissect'
    });
  }
  if (isManualIngestPipelineJsonProcessor(processor)) {
    const {
      manual_ingest_pipeline
    } = processor;
    return structuredClone({
      ...manual_ingest_pipeline,
      type: 'manual_ingest_pipeline'
    });
  }
  if (isDateProcessor(processor)) {
    const {
      date
    } = processor;
    return structuredClone({
      ...date,
      type: 'date'
    });
  }
  if (processor.type in _config_driven.configDrivenProcessors) {
    return _config_driven.configDrivenProcessors[processor.type].convertProcessorToFormState(processor);
  }
  throw new Error(`Form state for processor type "${processor.type}" is not implemented.`);
};
exports.getFormStateFrom = getFormStateFrom;
const convertFormStateToProcessor = formState => {
  if (formState.type === 'grok') {
    const {
      patterns,
      field,
      pattern_definitions,
      ignore_failure,
      ignore_missing
    } = formState;
    return {
      processorDefinition: {
        grok: {
          if: formState.if,
          patterns: patterns.map(pattern => pattern.getExpression()).filter(pattern => pattern !== undefined),
          field,
          pattern_definitions,
          ignore_failure,
          ignore_missing
        }
      },
      processorResources: {
        grokExpressions: patterns
      }
    };
  }
  if (formState.type === 'dissect') {
    const {
      field,
      pattern,
      append_separator,
      ignore_failure,
      ignore_missing
    } = formState;
    return {
      processorDefinition: {
        dissect: {
          if: formState.if,
          field,
          pattern,
          append_separator: (0, _lodash.isEmpty)(append_separator) ? undefined : append_separator,
          ignore_failure,
          ignore_missing
        }
      }
    };
  }
  if (formState.type === 'manual_ingest_pipeline') {
    const {
      processors,
      ignore_failure
    } = formState;
    return {
      processorDefinition: {
        manual_ingest_pipeline: {
          if: formState.if,
          processors,
          ignore_failure
        }
      }
    };
  }
  if (formState.type === 'date') {
    const {
      field,
      formats,
      locale,
      ignore_failure,
      target_field,
      timezone,
      output_format
    } = formState;
    return {
      processorDefinition: {
        date: {
          if: formState.if,
          field,
          formats,
          ignore_failure,
          locale: (0, _lodash.isEmpty)(locale) ? undefined : locale,
          target_field: (0, _lodash.isEmpty)(target_field) ? undefined : target_field,
          timezone: (0, _lodash.isEmpty)(timezone) ? undefined : timezone,
          output_format: (0, _lodash.isEmpty)(output_format) ? undefined : output_format
        }
      }
    };
  }
  if (_config_driven.configDrivenProcessors[formState.type]) {
    return {
      processorDefinition: _config_driven.configDrivenProcessors[formState.type].convertFormStateToConfig(formState)
    };
  }
  throw new Error('Cannot convert form state to processing: unknown type.');
};
exports.convertFormStateToProcessor = convertFormStateToProcessor;
const createProcessorGuardByType = type => processor => processor.type === type;
const isDateProcessor = exports.isDateProcessor = createProcessorGuardByType('date');
const isDissectProcessor = exports.isDissectProcessor = createProcessorGuardByType('dissect');
const isManualIngestPipelineJsonProcessor = exports.isManualIngestPipelineJsonProcessor = createProcessorGuardByType('manual_ingest_pipeline');
const isGrokProcessor = exports.isGrokProcessor = createProcessorGuardByType('grok');
const createId = (0, _eui.htmlIdGenerator)();
const processorToUIDefinition = processor => ({
  id: createId(),
  type: (0, _streamsSchema.getProcessorType)(processor),
  ...processor
});
const processorToAPIDefinition = processor => {
  const {
    id,
    type,
    ...processorConfig
  } = processor;
  return processorConfig;
};
const processorToSimulateDefinition = processor => {
  const {
    type,
    ...processorConfig
  } = processor;
  return processorConfig;
};
const processorConverter = exports.processorConverter = {
  toAPIDefinition: processorToAPIDefinition,
  toSimulateDefinition: processorToSimulateDefinition,
  toUIDefinition: processorToUIDefinition
};
const dataSourceToUIDefinition = dataSource => ({
  id: createId(),
  ...dataSource
});
const dataSourceToUrlSchema = dataSourceWithUIAttributes => {
  const {
    id,
    ...dataSource
  } = dataSourceWithUIAttributes;
  return dataSource;
};
const dataSourceConverter = exports.dataSourceConverter = {
  toUIDefinition: dataSourceToUIDefinition,
  toUrlSchema: dataSourceToUrlSchema
};
const recalcColumnWidths = ({
  columnId,
  width,
  prevWidths,
  visibleColumns
}) => {
  const next = {
    ...prevWidths
  };
  if (width === undefined) {
    delete next[columnId];
  } else {
    next[columnId] = width;
  }
  const allExplicit = visibleColumns.every(c => next[c] !== undefined);
  if (allExplicit) {
    delete next[visibleColumns[visibleColumns.length - 1]];
  }
  return next;
};
exports.recalcColumnWidths = recalcColumnWidths;