"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initDeleteSpacesApi = initDeleteSpacesApi;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _common = require("../../../../common");
var _errors = require("../../../lib/errors");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function initDeleteSpacesApi(deps) {
  const {
    router,
    log,
    getSpacesService
  } = deps;
  router.versioned.delete({
    path: '/api/spaces/space/{id}',
    access: 'public',
    summary: `Delete a space`,
    description: `When you delete a space, all saved objects that belong to the space are automatically deleted, which is permanent and cannot be undone.`,
    options: {
      tags: ['oas-tag:spaces']
    }
  }).addVersion({
    version: _common.API_VERSIONS.public.v1,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the spaces service via a scoped spaces client'
      }
    },
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'The space identifier.'
            }
          })
        })
      },
      response: {
        204: {
          description: 'Indicates a successful call.'
        },
        404: {
          description: 'Indicates that the request failed.'
        }
      }
    }
  }, (0, _lib.createLicensedRouteHandler)(async (context, request, response) => {
    const spacesClient = getSpacesService().createSpacesClient(request);
    const id = request.params.id;
    try {
      await spacesClient.delete(id);
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        return response.notFound();
      } else if (_server.SavedObjectsErrorHelpers.isEsCannotExecuteScriptError(error)) {
        log.error(`Failed to delete space '${id}', cannot execute script in Elasticsearch query: ${error.message}`);
        return response.customError((0, _errors.wrapError)(_boom.default.badRequest('Cannot execute script in Elasticsearch query')));
      }
      return response.customError((0, _errors.wrapError)(error));
    }
    return response.noContent();
  }));
}