"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getGlobalWeightForIdentifierType = exports.filterFromRange = exports.calculateRiskScores = void 0;
var _lodash = require("lodash");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _utils = require("../../../../common/entity_analytics/risk_engine/utils");
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _with_security_span = require("../../../utils/with_security_span");
var _helpers = require("../asset_criticality/helpers");
var _helpers2 = require("./helpers");
var _constants = require("./constants");
var _painless = require("./painless");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatForResponse = ({
  bucket,
  criticality,
  now,
  identifierField,
  includeNewFields
}) => {
  const riskDetails = bucket.top_inputs.risk_details;
  const criticalityModifier = (0, _helpers.getCriticalityModifier)(criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level);
  const normalizedScoreWithCriticality = (0, _helpers.applyCriticalityToScore)({
    score: riskDetails.value.normalized_score,
    modifier: criticalityModifier
  });
  const calculatedLevel = (0, _risk_engine.getRiskLevel)(normalizedScoreWithCriticality);
  const categoryTwoScore = normalizedScoreWithCriticality - riskDetails.value.normalized_score;
  const categoryTwoCount = criticalityModifier ? 1 : 0;
  const newFields = {
    category_2_score: categoryTwoScore,
    category_2_count: categoryTwoCount,
    criticality_level: criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level,
    criticality_modifier: criticalityModifier
  };
  return {
    '@timestamp': now,
    id_field: identifierField,
    id_value: bucket.key[identifierField],
    calculated_level: calculatedLevel,
    calculated_score: riskDetails.value.score,
    calculated_score_norm: normalizedScoreWithCriticality,
    category_1_score: riskDetails.value.category_1_score / _constants.RIEMANN_ZETA_VALUE,
    // normalize value to be between 0-100
    category_1_count: riskDetails.value.category_1_count,
    notes: riskDetails.value.notes,
    inputs: riskDetails.value.risk_inputs.map(riskInput => {
      var _riskInput$rule_name;
      return {
        id: riskInput.id,
        index: riskInput.index,
        description: `Alert from Rule: ${(_riskInput$rule_name = riskInput.rule_name) !== null && _riskInput$rule_name !== void 0 ? _riskInput$rule_name : 'RULE_NOT_FOUND'}`,
        category: _risk_engine.RiskCategories.category_1,
        risk_score: riskInput.score,
        timestamp: riskInput.time,
        contribution_score: riskInput.contribution
      };
    }),
    ...(includeNewFields ? newFields : {})
  };
};
const filterFromRange = range => ({
  range: {
    '@timestamp': {
      lt: range.end,
      gte: range.start
    }
  }
});
exports.filterFromRange = filterFromRange;
const buildIdentifierTypeAggregation = ({
  afterKeys,
  identifierType,
  pageSize,
  weights,
  alertSampleSizePerShard,
  scriptedMetricPainless
}) => {
  const globalIdentifierTypeWeight = getGlobalWeightForIdentifierType(identifierType, weights);
  const identifierField = (0, _helpers2.getFieldForIdentifier)(identifierType);
  return {
    composite: {
      size: pageSize,
      sources: [{
        [identifierField]: {
          terms: {
            field: identifierField
          }
        }
      }],
      after: (0, _helpers2.getAfterKeyForIdentifierType)({
        identifierType,
        afterKeys
      })
    },
    aggs: {
      top_inputs: {
        sampler: {
          shard_size: alertSampleSizePerShard
        },
        aggs: {
          risk_details: {
            scripted_metric: {
              init_script: scriptedMetricPainless.init,
              map_script: scriptedMetricPainless.map,
              combine_script: scriptedMetricPainless.combine,
              params: {
                p: _constants.RIEMANN_ZETA_S_VALUE,
                risk_cap: _constants.RIEMANN_ZETA_VALUE,
                global_identifier_type_weight: globalIdentifierTypeWeight || 1
              },
              reduce_script: scriptedMetricPainless.reduce
            }
          }
        }
      }
    }
  };
};
const processScores = async ({
  assetCriticalityService,
  buckets,
  identifierField,
  logger,
  now
}) => {
  if (buckets.length === 0) {
    return [];
  }
  const identifiers = buckets.map(bucket => ({
    id_field: identifierField,
    id_value: bucket.key[identifierField]
  }));
  let criticalities = [];
  try {
    criticalities = await assetCriticalityService.getCriticalitiesByIdentifiers(identifiers);
  } catch (e) {
    logger.warn(`Error retrieving criticality: ${e}. Scoring will proceed without criticality information.`);
  }
  return buckets.map(bucket => {
    const criticality = criticalities.find(c => c.id_field === identifierField && c.id_value === bucket.key[identifierField]);
    return formatForResponse({
      bucket,
      criticality,
      identifierField,
      now,
      includeNewFields: true
    });
  });
};
const getGlobalWeightForIdentifierType = (identifierType, weights) => {
  var _weights$find;
  return weights === null || weights === void 0 ? void 0 : (_weights$find = weights.find(weight => weight.type === _risk_engine.RiskWeightTypes.global)) === null || _weights$find === void 0 ? void 0 : _weights$find[identifierType];
};
exports.getGlobalWeightForIdentifierType = getGlobalWeightForIdentifierType;
const calculateRiskScores = async ({
  afterKeys: userAfterKeys,
  assetCriticalityService,
  debug,
  esClient,
  filter: userFilter,
  identifierType,
  index,
  logger,
  pageSize,
  range,
  runtimeMappings,
  weights,
  alertSampleSizePerShard = 10_000,
  experimentalFeatures,
  excludeAlertStatuses = [],
  excludeAlertTags = []
}) => (0, _with_security_span.withSecuritySpan)('calculateRiskScores', async () => {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4, _response$aggregation5, _response$aggregation6, _response$aggregation7, _response$aggregation8, _response$aggregation9;
  const now = new Date().toISOString();
  const scriptedMetricPainless = await (0, _painless.getPainlessScripts)();
  const filter = [filterFromRange(range), {
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }];
  if (excludeAlertStatuses.length > 0) {
    filter.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]: excludeAlertStatuses
          }
        }
      }
    });
  }
  if (!(0, _lodash.isEmpty)(userFilter)) {
    filter.push(userFilter);
  }
  if (excludeAlertTags.length > 0) {
    filter.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_TAGS]: excludeAlertTags
          }
        }
      }
    });
  }
  const identifierTypes = identifierType ? [identifierType] : (0, _utils.getRiskEngineEntityTypes)(experimentalFeatures);
  const request = {
    size: 0,
    _source: false,
    index,
    ignore_unavailable: true,
    runtime_mappings: runtimeMappings,
    query: {
      function_score: {
        query: {
          bool: {
            filter,
            should: [{
              match_all: {} // This forces ES to calculate score
            }]
          }
        },
        field_value_factor: {
          field: _technical_rule_data_field_names.ALERT_RISK_SCORE // sort by risk score
        }
      }
    },
    aggs: identifierTypes.reduce((aggs, _identifierType) => {
      aggs[_identifierType] = buildIdentifierTypeAggregation({
        afterKeys: userAfterKeys,
        identifierType: _identifierType,
        pageSize,
        weights,
        alertSampleSizePerShard,
        scriptedMetricPainless
      });
      return aggs;
    }, {})
  };
  if (debug) {
    logger.info(`Executing Risk Score query:\n${JSON.stringify(request)}`);
  }
  const response = await esClient.search(request);
  if (debug) {
    logger.info(`Received Risk Score response:\n${JSON.stringify(response)}`);
  }
  if (response.aggregations == null) {
    return {
      ...(debug ? {
        request,
        response
      } : {}),
      after_keys: {},
      scores: {
        host: [],
        user: [],
        service: []
      }
    };
  }
  const userBuckets = (_response$aggregation = (_response$aggregation2 = response.aggregations.user) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  const hostBuckets = (_response$aggregation3 = (_response$aggregation4 = response.aggregations.host) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.buckets) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : [];
  const serviceBuckets = (_response$aggregation5 = (_response$aggregation6 = response.aggregations.service) === null || _response$aggregation6 === void 0 ? void 0 : _response$aggregation6.buckets) !== null && _response$aggregation5 !== void 0 ? _response$aggregation5 : [];
  const afterKeys = {
    host: (_response$aggregation7 = response.aggregations.host) === null || _response$aggregation7 === void 0 ? void 0 : _response$aggregation7.after_key,
    user: (_response$aggregation8 = response.aggregations.user) === null || _response$aggregation8 === void 0 ? void 0 : _response$aggregation8.after_key,
    service: experimentalFeatures ? (_response$aggregation9 = response.aggregations.service) === null || _response$aggregation9 === void 0 ? void 0 : _response$aggregation9.after_key : undefined
  };
  const hostScores = await processScores({
    assetCriticalityService,
    buckets: hostBuckets,
    identifierField: 'host.name',
    logger,
    now
  });
  const userScores = await processScores({
    assetCriticalityService,
    buckets: userBuckets,
    identifierField: 'user.name',
    logger,
    now
  });
  const serviceScores = await processScores({
    assetCriticalityService,
    buckets: serviceBuckets,
    identifierField: 'service.name',
    logger,
    now
  });
  return {
    ...(debug ? {
      request,
      response
    } : {}),
    after_keys: afterKeys,
    scores: {
      host: hostScores,
      user: userScores,
      service: serviceScores
    }
  };
});
exports.calculateRiskScores = calculateRiskScores;