"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.eqlExecutor = void 0;
var _perf_hooks = require("perf_hooks");
var _isEmpty = _interopRequireDefault(require("lodash/isEmpty"));
var _build_eql_search_request = require("./build_eql_search_request");
var _utils = require("../utils/utils");
var _reason_formatters = require("../utils/reason_formatters");
var _with_security_span = require("../../../../utils/with_security_span");
var _bulk_create_suppressed_alerts_in_memory = require("../utils/bulk_create_suppressed_alerts_in_memory");
var _get_data_tier_filter = require("../utils/get_data_tier_filter");
var _logged_requests = require("../utils/logged_requests");
var i18n = _interopRequireWildcard(require("../translations"));
var _get_is_alert_suppression_active = require("../utils/get_is_alert_suppression_active");
var _utils2 = require("../../../../../common/detection_engine/utils");
var _log_shard_failure = require("../utils/log_shard_failure");
var _check_error_details = require("../utils/check_error_details");
var _wrap_sequences = require("./wrap_sequences");
var _factories = require("../factories");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const eqlExecutor = async ({
  sharedParams,
  services,
  wrapSuppressedHits,
  isAlertSuppressionActive,
  experimentalFeatures,
  state,
  scheduleNotificationResponseActionsService
}) => {
  var _state$isLoggedReques;
  const {
    completeRule,
    tuple,
    ruleExecutionLogger
  } = sharedParams;
  const ruleParams = completeRule.ruleParams;
  const isLoggedRequestsEnabled = (_state$isLoggedReques = state === null || state === void 0 ? void 0 : state.isLoggedRequestsEnabled) !== null && _state$isLoggedReques !== void 0 ? _state$isLoggedReques : false;
  const loggedRequests = [];

  // eslint-disable-next-line complexity
  return (0, _with_security_span.withSecuritySpan)('eqlExecutor', async () => {
    const result = (0, _utils.createSearchAfterReturnType)();
    const dataTiersFilters = await (0, _get_data_tier_filter.getDataTierFilter)({
      uiSettingsClient: services.uiSettingsClient
    });
    const isSequenceQuery = (0, _utils2.isEqlSequenceQuery)(ruleParams.query);
    const request = (0, _build_eql_search_request.buildEqlSearchRequest)({
      sharedParams,
      query: ruleParams.query,
      from: tuple.from.toISOString(),
      to: tuple.to.toISOString(),
      size: ruleParams.maxSignals,
      filters: [...(ruleParams.filters || []), ...dataTiersFilters],
      eventCategoryOverride: ruleParams.eventCategoryOverride,
      timestampField: ruleParams.timestampField,
      tiebreakerField: ruleParams.tiebreakerField
    });
    ruleExecutionLogger.debug(`EQL query request: ${JSON.stringify(request)}`);
    const exceptionsWarning = (0, _utils.getUnprocessedExceptionsWarnings)(sharedParams.unprocessedExceptions);
    if (exceptionsWarning) {
      result.warningMessages.push(exceptionsWarning);
    }
    const eqlSignalSearchStart = _perf_hooks.performance.now();
    try {
      var _newSignals;
      if (isLoggedRequestsEnabled) {
        loggedRequests.push({
          request: (0, _logged_requests.logEqlRequest)(request),
          description: i18n.EQL_SEARCH_REQUEST_DESCRIPTION
        });
      }
      const response = await services.scopedClusterClient.asCurrentUser.eql.search(request);
      const eqlSignalSearchEnd = _perf_hooks.performance.now();
      const eqlSearchDuration = eqlSignalSearchEnd - eqlSignalSearchStart;
      result.searchAfterTimes = [(0, _utils.makeFloatString)(eqlSearchDuration)];
      if (isLoggedRequestsEnabled && loggedRequests[0]) {
        loggedRequests[0].duration = Math.round(eqlSearchDuration);
      }
      let newSignals;
      const shardFailures = response.shard_failures;
      if (!(0, _isEmpty.default)(shardFailures)) {
        (0, _log_shard_failure.logShardFailures)(isSequenceQuery, shardFailures, result, ruleExecutionLogger);
      }
      const {
        events,
        sequences
      } = response.hits;
      if (events) {
        if (isAlertSuppressionActive && (0, _get_is_alert_suppression_active.alertSuppressionTypeGuard)(completeRule.ruleParams.alertSuppression)) {
          await (0, _bulk_create_suppressed_alerts_in_memory.bulkCreateSuppressedAlertsInMemory)({
            sharedParams,
            enrichedEvents: events,
            toReturn: result,
            services,
            buildReasonMessage: _reason_formatters.buildReasonMessageForEqlAlert,
            alertSuppression: completeRule.ruleParams.alertSuppression,
            wrapSuppressedHits
          });
        } else {
          newSignals = (0, _factories.wrapHits)(sharedParams, events, _reason_formatters.buildReasonMessageForEqlAlert);
        }
      } else if (sequences) {
        if (isAlertSuppressionActive && (0, _get_is_alert_suppression_active.alertSuppressionTypeGuard)(completeRule.ruleParams.alertSuppression)) {
          await (0, _bulk_create_suppressed_alerts_in_memory.bulkCreateSuppressedSequencesInMemory)({
            sharedParams,
            sequences,
            toReturn: result,
            services,
            buildReasonMessage: _reason_formatters.buildReasonMessageForEqlAlert,
            alertSuppression: completeRule.ruleParams.alertSuppression
          });
        } else {
          newSignals = (0, _wrap_sequences.wrapSequences)({
            sharedParams,
            sequences,
            buildReasonMessage: _reason_formatters.buildReasonMessageForEqlAlert
          });
        }
      } else {
        throw new Error('eql query response should have either `sequences` or `events` but had neither');
      }
      if ((_newSignals = newSignals) !== null && _newSignals !== void 0 && _newSignals.length) {
        const createResult = await (0, _factories.bulkCreate)({
          wrappedAlerts: newSignals,
          sharedParams,
          services
        });
        (0, _utils.addToSearchAfterReturn)({
          current: result,
          next: createResult
        });
      }
      if (response.hits.total && response.hits.total.value >= ruleParams.maxSignals) {
        const maxSignalsWarning = isAlertSuppressionActive && events !== null && events !== void 0 && events.length ? (0, _utils.getSuppressionMaxSignalsWarning)() : (0, _utils.getMaxSignalsWarning)();
        result.warningMessages.push(maxSignalsWarning);
      }
      scheduleNotificationResponseActionsService({
        signals: result.createdSignals,
        signalsCount: result.createdSignalsCount,
        responseActions: completeRule.ruleParams.responseActions
      });
      return {
        result,
        ...(isLoggedRequestsEnabled ? {
          loggedRequests
        } : {})
      };
    } catch (error) {
      if ((0, _check_error_details.checkErrorDetails)(error).isUserError) {
        result.userError = true;
      }
      result.errors.push(error.message);
      result.success = false;
      return {
        result,
        ...(isLoggedRequestsEnabled ? {
          loggedRequests
        } : {})
      };
    }
  });
};
exports.eqlExecutor = eqlExecutor;