"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineQueryRolesRoutes = defineQueryRolesRoutes;
var _configSchema = require("@kbn/config-schema");
var _coreSecurityServer = require("@kbn/core-security-server");
var _constants = require("../../../../common/constants");
var _authorization = require("../../../authorization");
var _errors = require("../../../errors");
var _licensed_route_handler = require("../../licensed_route_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function defineQueryRolesRoutes({
  router,
  authz,
  getFeatures,
  logger,
  buildFlavor
}) {
  router.versioned.post({
    path: '/api/security/role/_query',
    access: 'public',
    summary: `Query roles`,
    options: {
      tags: ['oas-tags:roles']
    },
    security: {
      authz: _coreSecurityServer.AuthzDisabled.delegateToESClient
    }
  }).addVersion({
    version: _constants.API_VERSIONS.roles.public.v1,
    validate: {
      request: {
        body: _configSchema.schema.object({
          query: _configSchema.schema.maybe(_configSchema.schema.string()),
          from: _configSchema.schema.maybe(_configSchema.schema.number()),
          size: _configSchema.schema.maybe(_configSchema.schema.number()),
          sort: _configSchema.schema.maybe(_configSchema.schema.object({
            field: _configSchema.schema.string(),
            direction: _configSchema.schema.oneOf([_configSchema.schema.literal('asc'), _configSchema.schema.literal('desc')])
          })),
          filters: _configSchema.schema.maybe(_configSchema.schema.object({
            showReservedRoles: _configSchema.schema.maybe(_configSchema.schema.boolean({
              defaultValue: true
            }))
          }))
        })
      },
      response: {
        200: {
          description: 'Indicates a successful call.'
        }
      }
    }
  }, (0, _licensed_route_handler.createLicensedRouteHandler)(async (context, request, response) => {
    try {
      const esClient = (await context.core).elasticsearch.client;
      const features = await getFeatures();
      const {
        query,
        size,
        from,
        sort,
        filters
      } = request.body;
      let showReservedRoles = filters === null || filters === void 0 ? void 0 : filters.showReservedRoles;
      if (buildFlavor === 'serverless') {
        showReservedRoles = false;
      }
      const queryPayload = {
        bool: {
          must: [],
          should: [],
          must_not: []
        }
      };
      const nonReservedRolesQuery = [{
        bool: {
          must_not: {
            exists: {
              field: 'metadata._reserved'
            }
          }
        }
      }];
      queryPayload.bool.should.push(...nonReservedRolesQuery);
      queryPayload.bool.minimum_should_match = 1;
      if (query) {
        queryPayload.bool.must.push({
          wildcard: {
            name: {
              value: `*${query}*`,
              case_insensitive: true
            }
          }
        });
      }
      if (showReservedRoles) {
        queryPayload.bool.should.push({
          term: {
            'metadata._reserved': true
          }
        });
      }
      const transformedSort = sort && [{
        [sort.field]: {
          order: sort.direction
        }
      }];
      const queryRoles = await esClient.asCurrentUser.security.queryRole({
        query: queryPayload,
        from,
        size,
        sort: transformedSort
      });
      const transformedRoles = (queryRoles.roles || []).map(role => (0, _authorization.transformElasticsearchRoleToRole)({
        features,
        // @ts-expect-error `remote_cluster` is not known in `Role` type
        elasticsearchRole: role,
        name: role.name,
        application: authz.applicationName,
        logger
      }));
      return response.ok({
        body: {
          roles: transformedRoles,
          count: queryRoles.count,
          total: queryRoles.total
        }
      });
    } catch (error) {
      return response.customError((0, _errors.wrapIntoCustomErrorResponse)(error));
    }
  }));
}