"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectConnectorStats = void 0;
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const collectConnectorStats = async client => {
  const connectors = await (0, _.fetchConnectors)(client);
  const syncJobs = [];
  let hasMore;
  let from = 0;
  do {
    const result = await (0, _.fetchSyncJobs)(client, undefined, from);
    syncJobs.push(...result.data);
    hasMore = result._meta.page.has_more_hits_than_total;
    from += result._meta.page.size;
  } while (hasMore);
  const connectorStatsArray = [];
  const syncJobsMap = groupSyncJobsByConnector(syncJobs);
  for (const connector of connectors) {
    var _connector$configurat, _connector$configurat2;
    // skip crawlers
    if (connector.service_type === _.CRAWLER_SERVICE_TYPE) {
      continue;
    }
    const connectorStats = {
      id: connector.id,
      serviceType: connector.service_type,
      isNative: connector.is_native,
      isDeleted: false,
      status: connector.status,
      indexName: connector.index_name,
      dlsEnabled: !!((_connector$configurat = connector.configuration.use_document_level_security) !== null && _connector$configurat !== void 0 && _connector$configurat.value),
      sslEnabled: connector.configuration.ssl_enabled ? !!connector.configuration.ssl_enabled.value : false,
      fetchSelectively: fetchSelectively(connector),
      textExtractionServiceEnabled: !!((_connector$configurat2 = connector.configuration.use_text_extraction_service) !== null && _connector$configurat2 !== void 0 && _connector$configurat2.value),
      documents: await documentsStats(client, connector),
      dataSourceSpecific: await dataSourceSpecificStats(client, connector),
      scheduling: {
        accessControl: connector.scheduling.access_control,
        full: connector.scheduling.full,
        incremental: connector.scheduling.incremental
      }
    };
    if (connector.pipeline) {
      connectorStats.ingestPipeline = {
        name: connector.pipeline.name,
        extractBinaryContent: connector.pipeline.extract_binary_content,
        reduceWhitespace: connector.pipeline.reduce_whitespace,
        runMLInference: connector.pipeline.run_ml_inference
      };
    }
    if (connector.filtering.length > 0) {
      var _filtering$active, _filtering$active2, _filtering$active2$ad, _filtering$draft, _filtering$draft2, _filtering$draft2$adv;
      const filtering = connector.filtering[0];
      connectorStats.syncRules = {
        active: {
          withBasicRules: Array.isArray(filtering === null || filtering === void 0 ? void 0 : (_filtering$active = filtering.active) === null || _filtering$active === void 0 ? void 0 : _filtering$active.rules) && filtering.active.rules.length > 1,
          withAdvancedRules: !!(filtering !== null && filtering !== void 0 && (_filtering$active2 = filtering.active) !== null && _filtering$active2 !== void 0 && (_filtering$active2$ad = _filtering$active2.advanced_snippet) !== null && _filtering$active2$ad !== void 0 && _filtering$active2$ad.value) && Object.keys(filtering.active.advanced_snippet.value).length > 0
        },
        draft: {
          withBasicRules: Array.isArray(filtering === null || filtering === void 0 ? void 0 : (_filtering$draft = filtering.draft) === null || _filtering$draft === void 0 ? void 0 : _filtering$draft.rules) && filtering.draft.rules.length > 1,
          withAdvancedRules: !!(filtering !== null && filtering !== void 0 && (_filtering$draft2 = filtering.draft) !== null && _filtering$draft2 !== void 0 && (_filtering$draft2$adv = _filtering$draft2.advanced_snippet) !== null && _filtering$draft2$adv !== void 0 && _filtering$draft2$adv.value) && Object.keys(filtering.draft.advanced_snippet.value).length > 0
        }
      };
    }
    if (syncJobsMap.has(connector.id)) {
      // @ts-ignore
      connectorStats.syncJobs = syncJobsStats(syncJobsMap.get(connector.id));
      syncJobsMap.delete(connector.id);
    }
    connectorStatsArray.push(connectorStats);
  }

  // process orphaned sync jobs
  for (const [connectorId, orphanedSyncJobs] of syncJobsMap) {
    const connectorStats = {
      id: connectorId,
      isDeleted: true,
      serviceType: orphanedSyncJobs[0].connector.service_type,
      syncJobs: syncJobsStats(orphanedSyncJobs)
    };
    connectorStatsArray.push(connectorStats);
  }
  return connectorStatsArray;
};
exports.collectConnectorStats = collectConnectorStats;
function groupSyncJobsByConnector(syncJobs) {
  const syncJobMaps = new Map();
  for (const syncJob of syncJobs) {
    // filter out sync jobs for crawler
    if (syncJob.connector.service_type === _.CRAWLER_SERVICE_TYPE) {
      continue;
    }
    const connectorId = syncJob.connector.id ? syncJob.connector.id : 'undefined';
    if (!syncJobMaps.has(connectorId)) {
      syncJobMaps.set(connectorId, []);
    }
    // @ts-ignore
    syncJobMaps.get(connectorId).push(syncJob);
  }
  return syncJobMaps;
}
function fetchSelectively(connector) {
  const rcfMap = {
    azure_blob_storage: 'containers',
    confluence: 'spaces',
    github: 'repositories',
    jira: 'projects',
    mssql: 'tables',
    mysql: 'tables',
    oracle: 'tables',
    postgresql: 'tables',
    s3: 'buckets',
    servicenow: 'services',
    sharepoint_online: 'site_collections',
    sharepoint_server: 'site_collections'
  };
  if (!connector.service_type || !(connector.service_type in rcfMap)) {
    return false;
  }
  const rcfField = rcfMap[connector.service_type];
  if (!(rcfField in connector.configuration)) {
    return false;
  }
  return !connector.configuration[rcfField].value.includes('*');
}
const documentsStats = async (client, connector) => {
  const stats = {
    total: 0,
    volume: 0,
    inLastSync: connector.last_indexed_document_count ? connector.last_indexed_document_count : 0
  };
  if (!connector.index_name) {
    return stats;
  }
  try {
    var _indicesStatsResponse, _indicesStatsResponse2, _indicesStatsResponse3, _indicesStatsResponse4, _indicesStatsResponse5, _indicesStatsResponse6;
    const indicesStatsResponse = await client.indices.stats({
      index: connector.index_name
    });
    stats.total = (_indicesStatsResponse = (_indicesStatsResponse2 = indicesStatsResponse._all.primaries) === null || _indicesStatsResponse2 === void 0 ? void 0 : (_indicesStatsResponse3 = _indicesStatsResponse2.docs) === null || _indicesStatsResponse3 === void 0 ? void 0 : _indicesStatsResponse3.count) !== null && _indicesStatsResponse !== void 0 ? _indicesStatsResponse : 0;
    stats.volume = (_indicesStatsResponse4 = (_indicesStatsResponse5 = indicesStatsResponse._all.primaries) === null || _indicesStatsResponse5 === void 0 ? void 0 : (_indicesStatsResponse6 = _indicesStatsResponse5.store) === null || _indicesStatsResponse6 === void 0 ? void 0 : _indicesStatsResponse6.size_in_bytes) !== null && _indicesStatsResponse4 !== void 0 ? _indicesStatsResponse4 : 0;
  } catch (e) {
    /* empty */
  }
  return stats;
};
const dataSourceSpecificStats = async (client, connector) => {
  var _connector$configurat3, _connector$configurat4, _connector$configurat5, _connector$configurat6, _connector$configurat7, _connector$configurat8, _connector$configurat9, _connector$configurat10, _connector$configurat11;
  const stats = {};
  switch (connector.service_type) {
    case 'confluence':
      stats.confluence = {
        dataSourceType: (_connector$configurat3 = connector.configuration.data_source) === null || _connector$configurat3 === void 0 ? void 0 : _connector$configurat3.value
      };
      break;
    case 'github':
      stats.github = {
        isCloud: ((_connector$configurat4 = connector.configuration.data_source) === null || _connector$configurat4 === void 0 ? void 0 : _connector$configurat4.value) === 'github_cloud'
      };
      break;
    case 'jira':
      stats.jira = {
        dataSourceType: (_connector$configurat5 = connector.configuration.data_source) === null || _connector$configurat5 === void 0 ? void 0 : _connector$configurat5.value
      };
      break;
    case 'mongodb':
      stats.mongodb = {
        directConnect: !!((_connector$configurat6 = connector.configuration.direct_connection) !== null && _connector$configurat6 !== void 0 && _connector$configurat6.value)
      };
      break;
    case 'mssql':
      stats.mssql = {
        validateHost: !!((_connector$configurat7 = connector.configuration.validate_host) !== null && _connector$configurat7 !== void 0 && _connector$configurat7.value),
        tables: connector.index_name ? await tableCounts(client, connector.index_name, 'table') : 0
      };
      break;
    case 'mysql':
      stats.mysql = {
        tables: connector.index_name ? await tableCounts(client, connector.index_name, 'Table') : 0
      };
      break;
    case 'oracle':
      stats.oracle = {
        tables: connector.index_name ? await tableCounts(client, connector.index_name, 'Table') : 0
      };
      break;
    case 'postgresql':
      stats.postgresql = {
        tables: connector.index_name ? await tableCounts(client, connector.index_name, 'table') : 0
      };
      break;
    case 'slack':
      stats.slack = {
        autoJoinChannelsEnabled: !!((_connector$configurat8 = connector.configuration.auto_join_channels) !== null && _connector$configurat8 !== void 0 && _connector$configurat8.value),
        syncUsersEnabled: !!((_connector$configurat9 = connector.configuration.sync_users) !== null && _connector$configurat9 !== void 0 && _connector$configurat9.value),
        fetchLastNDays: (_connector$configurat10 = connector.configuration.fetch_last_n_days) === null || _connector$configurat10 === void 0 ? void 0 : _connector$configurat10.value
      };
      break;
    case 'zoom':
      stats.zoom = {
        recordingAge: (_connector$configurat11 = connector.configuration.recording_age) === null || _connector$configurat11 === void 0 ? void 0 : _connector$configurat11.value
      };
      break;
  }
  return stats;
};
const tableCounts = async (client, indexName, tableField) => {
  try {
    var _value, _searchResponse$aggre;
    const aggs = {
      table_count: {
        cardinality: {
          field: `${tableField}.keyword`
        }
      }
    };
    const searchResponse = await client.search({
      index: indexName,
      aggs,
      size: 0
    });
    return (_value = ((_searchResponse$aggre = searchResponse.aggregations) === null || _searchResponse$aggre === void 0 ? void 0 : _searchResponse$aggre.table_count).value) !== null && _value !== void 0 ? _value : 0;
  } catch (e) {
    return 0;
  }
};
function syncJobsStats(syncJobs) {
  const stats = {
    overall: syncJobsStatsDetails(syncJobs)
  };
  const syncJobsWithTextExtractionServiceEnabled = syncJobs.filter(syncJob => {
    var _syncJob$connector$co;
    return !!((_syncJob$connector$co = syncJob.connector.configuration.use_text_extraction_service) !== null && _syncJob$connector$co !== void 0 && _syncJob$connector$co.value);
  });
  if (syncJobsWithTextExtractionServiceEnabled.length > 0) {
    stats.withTextExtractionServiceEnabled = syncJobsStatsDetails(syncJobsWithTextExtractionServiceEnabled);
  }
  return stats;
}
function syncJobsStatsDetails(syncJobs) {
  const stats = {
    total: syncJobs.length
  };
  const last30DaysSyncJobs = recentSyncJobs(30, syncJobs);
  if (last30DaysSyncJobs.length > 0) {
    stats.last30Days = syncJobsStatsByType(last30DaysSyncJobs);
  }
  const last7DaysSyncJobs = recentSyncJobs(7, syncJobs);
  if (last7DaysSyncJobs.length > 0) {
    stats.last7Days = syncJobsStatsByType(last7DaysSyncJobs);
  }
  return stats;
}
function recentSyncJobs(days, syncJobs) {
  const today = new Date();
  const nDaysAgo = new Date(today.setDate(today.getDate() - days));
  return syncJobs.filter(syncJob => {
    const createdAt = new Date(syncJob.created_at);
    return !isNaN(createdAt.getDate()) && createdAt > nDaysAgo;
  });
}
function syncJobsStatsByType(syncJobs) {
  const stats = {
    overall: syncJobsStatsByState(syncJobs)
  };
  const fullSyncJobs = syncJobs.filter(syncJob => syncJob.job_type === _.SyncJobType.FULL);
  if (fullSyncJobs.length > 0) {
    stats.full = syncJobsStatsByState(fullSyncJobs);
  }
  const incrementalSyncJobs = syncJobs.filter(syncJob => syncJob.job_type === _.SyncJobType.INCREMENTAL);
  if (incrementalSyncJobs.length > 0) {
    stats.incremental = syncJobsStatsByState(incrementalSyncJobs);
  }
  const accessControlSyncJobs = syncJobs.filter(syncJob => syncJob.job_type === _.SyncJobType.ACCESS_CONTROL);
  if (accessControlSyncJobs.length > 0) {
    stats.accessControl = syncJobsStatsByState(accessControlSyncJobs);
  }
  return stats;
}
function syncJobsStatsByState(syncJobs) {
  let manual = 0;
  let scheduled = 0;
  let completed = 0;
  let errored = 0;
  let canceled = 0;
  let suspended = 0;
  let idle = 0;
  let running = 0;
  let duration = 0;
  const errors = new Map();
  let topErrors = [];
  for (const syncJob of syncJobs) {
    completed += syncJob.status === _.SyncStatus.COMPLETED ? 1 : 0;
    errored += syncJob.status === _.SyncStatus.ERROR ? 1 : 0;
    canceled += syncJob.status === _.SyncStatus.CANCELED ? 1 : 0;
    suspended += syncJob.status === _.SyncStatus.SUSPENDED ? 1 : 0;
    running += syncJob.status === _.SyncStatus.IN_PROGRESS ? 1 : 0;
    manual += syncJob.trigger_method === _.TriggerMethod.ON_DEMAND ? 1 : 0;
    scheduled += syncJob.trigger_method === _.TriggerMethod.SCHEDULED ? 1 : 0;
    if (syncJob.status in [_.SyncStatus.IN_PROGRESS, _.SyncStatus.CANCELING] && syncJob.last_seen) {
      const lastSeen = new Date(syncJob.last_seen);
      // A sync job with last_seen not updated for more than 5 mins is considered idle
      if (!isNaN(lastSeen.getTime()) && new Date().getTime() - lastSeen.getTime() > 5 * 60 * 1000) {
        idle += 1;
      }
    }
    if (syncJob.started_at && syncJob.completed_at) {
      const startedAt = new Date(syncJob.started_at);
      const completedAt = new Date(syncJob.completed_at);
      if (!isNaN(startedAt.getTime()) && !isNaN(completedAt.getTime())) {
        duration += Math.floor((completedAt.getTime() - startedAt.getTime()) / 1000);
      }
    }
    if (syncJob.status === _.SyncStatus.ERROR && syncJob.error) {
      var _errors$get;
      errors.set(syncJob.error, ((_errors$get = errors.get(syncJob.error)) !== null && _errors$get !== void 0 ? _errors$get : 0) + 1);
    }
  }
  if (errors.size <= 5) {
    topErrors = [...errors.keys()];
  } else {
    topErrors = [...errors.entries()].sort((a, b) => b[1] - a[1]).map(a => a[0]).slice(0, 5);
  }
  return {
    total: syncJobs.length,
    manual,
    scheduled,
    completed,
    errored,
    canceled,
    suspended,
    idle,
    running,
    totalDurationSeconds: duration,
    topErrors
  };
}