"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMetricChanges = getMetricChanges;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getMetricAggregation({
  field,
  type
}) {
  if (type === 'count') {
    return field ? {
      value_count: {
        field
      }
    } : {
      filter: {
        match_all: {}
      },
      bucket_path: '_count'
    };
  }
  if (!field) {
    throw new Error(`Metric type ${type} needs a field to aggregate over`);
  }
  if (type === 'min' || 'max' || 'sum' || 'avg') {
    return {
      [type]: {
        field
      }
    };
  }
  const percentile = `${type.split('p')[1]}.0`;
  return {
    percentiles: {
      field,
      percents: [Number(percentile)],
      keyed: true
    },
    bucket_path: percentile
  };
}
function getGroupingAggregation(groupingFields) {
  if (groupingFields.length === 0) {
    return {
      filters: {
        filters: [{
          match_all: {}
        }]
      }
    };
  }
  if (groupingFields.length === 1) {
    return {
      terms: {
        field: groupingFields[0]
      }
    };
  }
  return {
    multi_terms: {
      terms: groupingFields.map(groupingField => ({
        field: groupingField
      })),
      size: 10
    }
  };
}
async function getMetricChanges({
  index,
  filters,
  groupBy,
  field,
  type,
  client,
  dateHistogram
}) {
  var _response$aggregation;
  const metricAgg = getMetricAggregation({
    type,
    field
  });
  const groupAgg = getGroupingAggregation(groupBy);
  const subAggs = {
    over_time: {
      auto_date_histogram: dateHistogram,
      aggs: {
        metric: metricAgg,
        value: {
          bucket_script: {
            buckets_path: {
              metric: `metric${'buckets_path' in metricAgg ? `>${metricAgg.buckets_path}` : ''}`
            },
            script: 'params.metric'
          }
        }
      }
    },
    changes: {
      change_point: {
        buckets_path: 'over_time>value'
      }
    }
  };
  const response = await client.search('get_metric_changes', {
    index,
    size: 0,
    track_total_hits: false,
    query: {
      bool: {
        filter: filters
      }
    },
    aggregations: {
      groups: {
        ...groupAgg,
        aggs: subAggs
      }
    }
  });
  const groups = ((_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation.groups.buckets) || [];
  const series = groups.map(group => {
    var _group$key, _group$changes$bucket;
    const key = (_group$key = group.key) !== null && _group$key !== void 0 ? _group$key : 'all';
    return {
      key,
      over_time: group.over_time.buckets.map(bucket => {
        var _bucket$value;
        return {
          x: new Date(bucket.key_as_string).getTime(),
          y: (_bucket$value = bucket.value) === null || _bucket$value === void 0 ? void 0 : _bucket$value.value
        };
      }),
      changes: group.changes.type.indeterminable || !((_group$changes$bucket = group.changes.bucket) !== null && _group$changes$bucket !== void 0 && _group$changes$bucket.key) ? {
        type: 'indeterminable'
      } : {
        time: new Date(group.changes.bucket.key).toISOString(),
        type: Object.keys(group.changes.type)[0],
        ...Object.values(group.changes.type)[0]
      }
    };
  });
  return series;
}