"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getInferenceServicesRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _common = require("../../common");
var _inference_endpoint_exists = require("../lib/inference_endpoint_exists");
var _unflatten_object = require("../utils/unflatten_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const inferenceEndpointSchema = _configSchema.schema.object({
  config: _configSchema.schema.object({
    inferenceId: _configSchema.schema.string(),
    provider: _configSchema.schema.string(),
    taskType: _configSchema.schema.string(),
    providerConfig: _configSchema.schema.any()
  }),
  secrets: _configSchema.schema.object({
    providerSecrets: _configSchema.schema.any()
  })
});
const getInferenceServicesRoute = (router, logger) => {
  router.versioned.get({
    access: 'internal',
    path: '/internal/_inference/_services'
  }).addVersion({
    version: _common.INFERENCE_ENDPOINT_INTERNAL_API_VERSION,
    validate: {}
  }, async (context, request, response) => {
    try {
      const esClient = (await context.core).elasticsearch.client.asInternalUser;
      const result = await esClient.transport.request({
        method: 'GET',
        path: `/_inference/_services`
      });
      return response.ok({
        body: result
      });
    } catch (err) {
      logger.error(err);
      return response.customError({
        body: {
          message: err.message
        },
        statusCode: err.statusCode
      });
    }
  });
  router.versioned.post({
    access: 'internal',
    path: '/internal/_inference/_add'
  }).addVersion({
    version: _common.INFERENCE_ENDPOINT_INTERNAL_API_VERSION,
    validate: {
      request: {
        body: inferenceEndpointSchema
      }
    }
  }, async (context, request, response) => {
    try {
      var _config$providerConfi, _config$providerConfi2, _secrets$providerSecr, _config$inferenceId;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const {
        config,
        secrets
      } = request.body;

      // NOTE: This is a temporary workaround for anthropic max_tokens handling until the services endpoint is updated to reflect the correct structure.
      // Anthropic is unique in that it requires max_tokens to be sent as part of the task_settings instead of the usual service_settings.
      // Until the services endpoint is updated to reflect that, there is no way for the form UI to know where to put max_tokens. This can be removed once that update is made.
      let taskSettings;
      if ((config === null || config === void 0 ? void 0 : config.provider) === 'anthropic' && config !== null && config !== void 0 && (_config$providerConfi = config.providerConfig) !== null && _config$providerConfi !== void 0 && _config$providerConfi.max_tokens) {
        taskSettings = {
          max_tokens: config.providerConfig.max_tokens
        };
        // This field is unknown to the anthropic service config, so we remove it
        delete config.providerConfig.max_tokens;
      }
      const serviceSettings = {
        ...(0, _unflatten_object.unflattenObject)((_config$providerConfi2 = config === null || config === void 0 ? void 0 : config.providerConfig) !== null && _config$providerConfi2 !== void 0 ? _config$providerConfi2 : {}),
        ...(0, _unflatten_object.unflattenObject)((_secrets$providerSecr = secrets === null || secrets === void 0 ? void 0 : secrets.providerSecrets) !== null && _secrets$providerSecr !== void 0 ? _secrets$providerSecr : {})
      };
      const result = await esClient.inference.put({
        inference_id: (_config$inferenceId = config === null || config === void 0 ? void 0 : config.inferenceId) !== null && _config$inferenceId !== void 0 ? _config$inferenceId : '',
        task_type: config === null || config === void 0 ? void 0 : config.taskType,
        inference_config: {
          service: config === null || config === void 0 ? void 0 : config.provider,
          service_settings: serviceSettings,
          ...(taskSettings ? {
            task_settings: taskSettings
          } : {})
        }
      });
      return response.ok({
        body: result
      });
    } catch (err) {
      logger.error(err);
      return response.customError({
        body: err.message,
        statusCode: err.statusCode
      });
    }
  });
  router.versioned.get({
    access: 'internal',
    path: '/internal/_inference/_exists/{inferenceId}'
  }).addVersion({
    version: _common.INFERENCE_ENDPOINT_INTERNAL_API_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          inferenceId: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    try {
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const result = await (0, _inference_endpoint_exists.inferenceEndpointExists)(esClient, request.params.inferenceId);
      return response.ok({
        body: {
          isEndpointExists: result
        }
      });
    } catch (err) {
      logger.error(err);
      return response.customError({
        body: err.message,
        statusCode: err.statusCode
      });
    }
  });
  router.versioned.put({
    access: 'internal',
    path: '/internal/_inference/_update'
  }).addVersion({
    version: _common.INFERENCE_ENDPOINT_INTERNAL_API_VERSION,
    validate: {
      request: {
        body: inferenceEndpointSchema
      }
    }
  }, async (context, request, response) => {
    try {
      var _secrets$providerSecr2, _config$providerConfi3;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const {
        config,
        secrets
      } = request.body;

      // currently update api only allows api_key and num_allocations
      const serviceSettings = {
        service_settings: {
          ...((secrets === null || secrets === void 0 ? void 0 : (_secrets$providerSecr2 = secrets.providerSecrets) === null || _secrets$providerSecr2 === void 0 ? void 0 : _secrets$providerSecr2.api_key) && {
            api_key: secrets.providerSecrets.api_key
          }),
          ...((config === null || config === void 0 ? void 0 : (_config$providerConfi3 = config.providerConfig) === null || _config$providerConfi3 === void 0 ? void 0 : _config$providerConfi3.num_allocations) !== undefined && {
            num_allocations: config.providerConfig.num_allocations
          })
        }
      };
      const result = await esClient.transport.request({
        method: 'PUT',
        path: `/_inference/${config.taskType}/${config.inferenceId}/_update`,
        body: JSON.stringify(serviceSettings)
      }, {
        headers: {
          'content-type': 'application/json'
        }
      });
      return response.ok({
        body: result
      });
    } catch (err) {
      logger.error(err);
      return response.customError({
        body: err.message,
        statusCode: err.statusCode
      });
    }
  });
};
exports.getInferenceServicesRoute = getInferenceServicesRoute;