"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useOverallStats = useOverallStats;
var _react = require("react");
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _mlDataGrid = require("@kbn/ml-data-grid");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _kibana_context = require("../../kibana_context");
var _overall_stats = require("../search_strategy/requests/overall_stats");
var _field_stats = require("../../../../common/types/field_stats");
var _get_document_stats = require("../search_strategy/requests/get_document_stats");
var _progress_utils = require("../progress_utils");
var _index_data_visualizer_viewer = require("../constants/index_data_visualizer_viewer");
var _display_error = require("../../common/util/display_error");
var _fetch_utils = require("../search_strategy/requests/fetch_utils");
var _build_query_filters = require("../../../../common/utils/build_query_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPopulatedFieldsInIndex = (populatedFieldsInIndexWithoutRuntimeFields, runtimeFieldMap) => {
  if (!populatedFieldsInIndexWithoutRuntimeFields) return undefined;
  const runtimeFields = runtimeFieldMap ? Object.keys(runtimeFieldMap) : undefined;
  return runtimeFields && (runtimeFields === null || runtimeFields === void 0 ? void 0 : runtimeFields.length) > 0 ? new Set([...Array.from(populatedFieldsInIndexWithoutRuntimeFields), ...runtimeFields]) : populatedFieldsInIndexWithoutRuntimeFields;
};
function useOverallStats(esql = false, searchStrategyParams, lastRefresh, probability) {
  const {
    services: {
      data,
      notifications: {
        toasts
      }
    }
  } = (0, _kibana_context.useDataVisualizerKibana)();
  const [stats, setOverallStats] = (0, _react.useState)((0, _index_data_visualizer_viewer.getDefaultPageState)().overallStats);
  const [populatedFieldsInIndexWithoutRuntimeFields, setPopulatedFieldsInIndex] = (0, _react.useState)();
  const [fetchState, setFetchState] = (0, _react.useReducer)((0, _progress_utils.getReducer)(), (0, _progress_utils.getInitialProgress)());
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const populatedFieldsAbortCtrl = (0, _react.useRef)(new AbortController());
  const searchSubscription$ = (0, _react.useRef)();
  (0, _react.useEffect)(function updatePopulatedFields() {
    let unmounted = false;

    // If null, that means we tried to fetch populated fields already but it timed out
    // so don't try again
    if (!searchStrategyParams || populatedFieldsInIndexWithoutRuntimeFields === null) return;
    const {
      index,
      searchQuery,
      timeFieldName,
      earliest,
      latest
    } = searchStrategyParams;
    const fetchPopulatedFields = async () => {
      populatedFieldsAbortCtrl.current.abort();
      populatedFieldsAbortCtrl.current = new AbortController();

      // Trick to avoid duplicate getFieldsForWildcard requests
      // wouldn't make sense to make time-based query if either earliest & latest timestamps is undefined
      if (timeFieldName !== undefined && (earliest === undefined || latest === undefined)) {
        return;
      }
      const filterCriteria = (0, _build_query_filters.buildFilterCriteria)(timeFieldName, earliest, latest, searchQuery);

      // Getting non-empty fields for the index pattern
      // because then we can absolutely exclude these from subsequent requests
      const nonEmptyFields = await (0, _fetch_utils.fetchDataWithTimeout)(data.dataViews.getFieldsForWildcard({
        pattern: index,
        indexFilter: {
          bool: {
            filter: filterCriteria
          }
        },
        includeEmptyFields: false
      }), populatedFieldsAbortCtrl.current);
      if (!unmounted) {
        if (Array.isArray(nonEmptyFields)) {
          setPopulatedFieldsInIndex(new Set([...nonEmptyFields.map(field => field.name)]));
        } else {
          setPopulatedFieldsInIndex(null);
        }
      }
    };
    fetchPopulatedFields();
    return () => {
      unmounted = true;
    };
  },
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [searchStrategyParams === null || searchStrategyParams === void 0 ? void 0 : searchStrategyParams.timeFieldName, searchStrategyParams === null || searchStrategyParams === void 0 ? void 0 : searchStrategyParams.earliest, searchStrategyParams === null || searchStrategyParams === void 0 ? void 0 : searchStrategyParams.latest,
  // eslint-disable-next-line react-hooks/exhaustive-deps
  JSON.stringify({
    query: searchStrategyParams === null || searchStrategyParams === void 0 ? void 0 : searchStrategyParams.searchQuery
  }), searchStrategyParams === null || searchStrategyParams === void 0 ? void 0 : searchStrategyParams.index]);
  const startFetch = (0, _react.useCallback)(async () => {
    try {
      var _searchSubscription$$;
      (_searchSubscription$$ = searchSubscription$.current) === null || _searchSubscription$$ === void 0 ? void 0 : _searchSubscription$$.unsubscribe();
      abortCtrl.current.abort();
      abortCtrl.current = new AbortController();
      if (!searchStrategyParams || lastRefresh === 0 || populatedFieldsInIndexWithoutRuntimeFields === undefined) {
        return;
      }
      const populatedFieldsInIndex = getPopulatedFieldsInIndex(populatedFieldsInIndexWithoutRuntimeFields, searchStrategyParams.runtimeFieldMap);
      setFetchState({
        ...(0, _progress_utils.getInitialProgress)(),
        isRunning: true,
        error: undefined
      });
      const {
        aggregatableFields: originalAggregatableFields,
        nonAggregatableFields: originalNonAggregatableFields,
        index,
        searchQuery,
        timeFieldName,
        earliest,
        latest,
        runtimeFieldMap,
        samplingOption,
        sessionId,
        embeddableExecutionContext
      } = searchStrategyParams;
      const searchOptions = {
        abortSignal: abortCtrl.current.signal,
        sessionId,
        ...(embeddableExecutionContext ? {
          executionContext: embeddableExecutionContext
        } : {})
      };
      const hasPopulatedFieldsInfo = (0, _mlIsDefined.isDefined)(populatedFieldsInIndex);
      const aggregatableFields = hasPopulatedFieldsInfo ? originalAggregatableFields.filter(field => populatedFieldsInIndex.has(field.name)) : originalAggregatableFields;
      const nonAggregatableFields = hasPopulatedFieldsInfo ? originalNonAggregatableFields.filter(fieldName => populatedFieldsInIndex.has(fieldName)) : originalNonAggregatableFields;
      const documentCountStats = await (0, _get_document_stats.getDocumentCountStats)(data.search, searchStrategyParams, searchOptions, samplingOption.seed, probability);
      const nonAggregatableFieldsExamplesObs = data.search.search({
        params: (0, _overall_stats.getSampleOfDocumentsForNonAggregatableFields)(nonAggregatableFields, index, searchQuery, timeFieldName, earliest, latest, runtimeFieldMap)
      }, searchOptions).pipe((0, _rxjs.map)(resp => {
        return resp;
      }));
      const nonAggregatableFieldsObs = nonAggregatableFields.map(fieldName => data.search.search({
        params: (0, _overall_stats.checkNonAggregatableFieldExistsRequest)(index, searchQuery, fieldName, timeFieldName, earliest, latest, runtimeFieldMap)
      }, searchOptions).pipe((0, _rxjs.map)(resp => {
        return {
          ...resp,
          rawResponse: {
            ...resp.rawResponse,
            fieldName
          }
        };
      })));

      // Have to divide into smaller requests to avoid 413 payload too large
      const aggregatableFieldsChunks = (0, _lodash.chunk)(aggregatableFields, 30);
      if ((0, _field_stats.isRandomSamplingOption)(samplingOption)) {
        var _documentCountStats$p;
        samplingOption.probability = (_documentCountStats$p = documentCountStats.probability) !== null && _documentCountStats$p !== void 0 ? _documentCountStats$p : 1;
      }
      const aggregatableOverallStatsObs = aggregatableFieldsChunks.map(aggregatableFieldsChunk => data.search.search({
        params: (0, _overall_stats.checkAggregatableFieldsExistRequest)(index, searchQuery, aggregatableFieldsChunk, samplingOption, timeFieldName, earliest, latest, undefined, runtimeFieldMap)
      }, searchOptions).pipe((0, _rxjs.map)(resp => {
        return {
          ...resp,
          aggregatableFields: aggregatableFieldsChunk
        };
      })));
      const sub = (0, _fetch_utils.rateLimitingForkJoin)([nonAggregatableFieldsExamplesObs, ...aggregatableOverallStatsObs, ...nonAggregatableFieldsObs], _index_data_visualizer_viewer.MAX_CONCURRENT_REQUESTS);
      searchSubscription$.current = sub.subscribe({
        next: value => {
          var _documentCountStats$t;
          const aggregatableOverallStatsResp = [];
          const nonAggregatableOverallStatsResp = [];
          let sampledNonAggregatableFieldsExamples;
          value.forEach((resp, idx) => {
            if (idx === 0 && (0, _overall_stats.isNonAggregatableSampledDocs)(resp)) {
              const docs = resp.rawResponse.hits.hits.map(d => d.fields ? (0, _mlDataGrid.getProcessedFields)(d.fields) : {});
              sampledNonAggregatableFieldsExamples = docs;
            }
            if ((0, _overall_stats.isAggregatableFieldOverallStats)(resp)) {
              aggregatableOverallStatsResp.push(resp);
            }
            if ((0, _overall_stats.isNonAggregatableFieldOverallStats)(resp)) {
              nonAggregatableOverallStatsResp.push(resp);
            }
          });
          const totalCount = (_documentCountStats$t = documentCountStats === null || documentCountStats === void 0 ? void 0 : documentCountStats.totalCount) !== null && _documentCountStats$t !== void 0 ? _documentCountStats$t : 0;
          const aggregatableOverallStats = (0, _overall_stats.processAggregatableFieldsExistResponse)(aggregatableOverallStatsResp, originalAggregatableFields, populatedFieldsInIndex);
          const nonAggregatableFieldsCount = new Array(nonAggregatableFields.length).fill(0);
          const nonAggregatableFieldsUniqueCount = nonAggregatableFields.map(() => new Set());
          if (sampledNonAggregatableFieldsExamples) {
            sampledNonAggregatableFieldsExamples.forEach(doc => {
              nonAggregatableFields.forEach((field, fieldIdx) => {
                if (Object.hasOwn(doc, field)) {
                  nonAggregatableFieldsCount[fieldIdx] += 1;
                  nonAggregatableFieldsUniqueCount[fieldIdx].add(doc[field]);
                }
              });
            });
          }
          const nonAggregatableOverallStats = (0, _overall_stats.processNonAggregatableFieldsExistResponse)(nonAggregatableOverallStatsResp, originalNonAggregatableFields, nonAggregatableFieldsCount, nonAggregatableFieldsUniqueCount, nonAggregatableFields);
          setOverallStats({
            documentCountStats,
            ...nonAggregatableOverallStats,
            ...aggregatableOverallStats,
            totalCount
          });
        },
        error: error => {
          if (error.name !== 'AbortError') {
            (0, _display_error.displayError)(toasts, searchStrategyParams.index, (0, _mlErrorUtils.extractErrorProperties)(error));
          }
          setFetchState({
            isRunning: false,
            error
          });
        },
        complete: () => {
          setFetchState({
            loaded: 100,
            isRunning: false
          });
        }
      });
    } catch (error) {
      // An `AbortError` gets triggered when a user cancels a request by navigating away, we need to ignore these errors.
      if (error.name !== 'AbortError') {
        (0, _display_error.displayError)(toasts, searchStrategyParams.index, (0, _mlErrorUtils.extractErrorProperties)(error));
      }
    }
  }, [data, searchStrategyParams, toasts, lastRefresh, probability, populatedFieldsInIndexWithoutRuntimeFields]);
  const cancelFetch = (0, _react.useCallback)(() => {
    var _searchSubscription$$2, _abortCtrl$current, _populatedFieldsAbort;
    (_searchSubscription$$2 = searchSubscription$.current) === null || _searchSubscription$$2 === void 0 ? void 0 : _searchSubscription$$2.unsubscribe();
    searchSubscription$.current = undefined;
    (_abortCtrl$current = abortCtrl.current) === null || _abortCtrl$current === void 0 ? void 0 : _abortCtrl$current.abort();
    (_populatedFieldsAbort = populatedFieldsAbortCtrl.current) === null || _populatedFieldsAbort === void 0 ? void 0 : _populatedFieldsAbort.abort();
  }, []);

  // auto-update
  (0, _react.useEffect)(() => {
    startFetch();
  }, [startFetch]);
  (0, _react.useEffect)(() => {
    return cancelFetch;
  }, [cancelFetch]);
  return (0, _react.useMemo)(() => ({
    progress: fetchState,
    overallStats: stats
  }), [stats, fetchState]);
}