"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.routeHandler = void 0;
var _transform = require("../../../../common/types/transform");
var _field_utils = require("../../../../common/utils/field_utils");
var _error_utils = require("../../utils/error_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const routeHandler = async (ctx, req, res) => {
  try {
    const reqBody = req.body;
    const esClient = (await ctx.core).elasticsearch.client;
    const body = await esClient.asCurrentUser.transform.previewTransform({
      body: reqBody
    }, {
      maxRetries: 0
    });
    if ((0, _transform.isLatestTransform)(reqBody)) {
      // for the latest transform mappings properties have to be retrieved from the source
      const fieldCapsResponse = await esClient.asCurrentUser.fieldCaps({
        index: reqBody.source.index,
        fields: '*',
        include_unmapped: false
      }, {
        maxRetries: 0
      });
      const fieldNamesSet = new Set(Object.keys(fieldCapsResponse.fields));
      const fields = Object.entries(fieldCapsResponse.fields).reduce((acc, [fieldName, fieldCaps]) => {
        const fieldDefinition = Object.values(fieldCaps)[0];
        const isMetaField = fieldDefinition.type.startsWith('_') || fieldName === '_doc_count';
        if (isMetaField || (0, _field_utils.isKeywordDuplicate)(fieldName, fieldNamesSet)) {
          return acc;
        }
        acc[fieldName] = {
          ...fieldDefinition
        };
        return acc;
      }, {});
      body.generated_dest_index.mappings.properties = fields;
    }
    return res.ok({
      body
    });
  } catch (e) {
    return res.customError((0, _error_utils.wrapError)((0, _error_utils.wrapEsError)(e)));
  }
};
exports.routeHandler = routeHandler;