"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mixParamsWithGlobalParams = exports.formatMonitorConfigFields = exports.formatHeartbeatRequest = void 0;
var _lodash = require("lodash");
var _formatting_utils = require("../formatting_utils");
var _common = require("../common");
var _runtime_types = require("../../../../common/runtime_types");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const UI_KEYS_TO_SKIP = [_runtime_types.ConfigKey.JOURNEY_ID, _runtime_types.ConfigKey.PROJECT_ID, _runtime_types.ConfigKey.METADATA, _runtime_types.ConfigKey.REVISION, _runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID, _runtime_types.ConfigKey.FORM_MONITOR_TYPE, _runtime_types.ConfigKey.TEXT_ASSERTION, _runtime_types.ConfigKey.CONFIG_HASH, _runtime_types.ConfigKey.ALERT_CONFIG, _runtime_types.ConfigKey.LABELS, 'secrets'];
const formatMonitorConfigFields = (configKeys, config, logger, params, mws) => {
  var _config$ConfigKey$MET;
  const formattedMonitor = {};
  configKeys.forEach(key => {
    if (!UI_KEYS_TO_SKIP.includes(key)) {
      var _config$key;
      const value = (_config$key = config[key]) !== null && _config$key !== void 0 ? _config$key : null;
      if (value === null || value === '') {
        return;
      }
      if (config.type !== 'browser' && key === _runtime_types.ConfigKey.PARAMS) {
        return;
      }
      if (!!_.publicFormatters[key]) {
        const formatter = _.publicFormatters[key];
        if (typeof formatter === 'function') {
          formattedMonitor[key] = formatter(config, key);
        } else {
          formattedMonitor[key] = formatter;
        }
      } else {
        formattedMonitor[key] = value;
      }
    }
    if (!_common.PARAMS_KEYS_TO_SKIP.includes(key)) {
      formattedMonitor[key] = (0, _formatting_utils.replaceStringWithParams)(formattedMonitor[key], params, logger);
    }
  });
  if (!((_config$ConfigKey$MET = config[_runtime_types.ConfigKey.METADATA]) !== null && _config$ConfigKey$MET !== void 0 && _config$ConfigKey$MET.is_tls_enabled)) {
    const sslKeys = Object.keys(formattedMonitor).filter(key => key.includes('ssl'));
    sslKeys.forEach(key => formattedMonitor[key] = null);
  }
  if (config[_runtime_types.ConfigKey.MAINTENANCE_WINDOWS]) {
    const maintenanceWindows = config[_runtime_types.ConfigKey.MAINTENANCE_WINDOWS];
    formattedMonitor[_runtime_types.ConfigKey.MAINTENANCE_WINDOWS] = (0, _formatting_utils.formatMWs)(maintenanceWindows.map(window => {
      if (typeof window === 'string') {
        return mws.find(m => m.id === window);
      }
      return window;
    }), false);
  }
  return (0, _lodash.omitBy)(formattedMonitor, _lodash.isNil);
};
exports.formatMonitorConfigFields = formatMonitorConfigFields;
const formatHeartbeatRequest = ({
  monitor,
  configId,
  heartbeatId,
  runOnce,
  testRunId,
  spaceId
}, params) => {
  const projectId = monitor[_runtime_types.ConfigKey.PROJECT_ID];
  const heartbeatIdT = heartbeatId !== null && heartbeatId !== void 0 ? heartbeatId : monitor[_runtime_types.ConfigKey.MONITOR_QUERY_ID];
  const paramsString = params !== null && params !== void 0 ? params : monitor[_runtime_types.ConfigKey.PARAMS];
  const {
    labels,
    spaces
  } = monitor;
  const monSpaces = spaces ? Array.from(new Set([...(spaces !== null && spaces !== void 0 ? spaces : []), spaceId])) : spaceId;
  return {
    ...monitor,
    id: heartbeatIdT,
    fields: {
      config_id: configId,
      'monitor.project.name': projectId || undefined,
      'monitor.project.id': projectId || undefined,
      run_once: runOnce,
      test_run_id: testRunId,
      meta: {
        space_id: monSpaces
      },
      ...((0, _lodash.isEmpty)(labels) ? {} : {
        labels
      })
    },
    fields_under_root: true,
    params: monitor.type === 'browser' ? paramsString : ''
  };
};
exports.formatHeartbeatRequest = formatHeartbeatRequest;
const mixParamsWithGlobalParams = (globalParams, monitor) => {
  let params = {
    ...(globalParams !== null && globalParams !== void 0 ? globalParams : {})
  };
  const paramsString = '';
  try {
    const monParamsStr = monitor[_runtime_types.ConfigKey.PARAMS];
    if (monParamsStr) {
      const monitorParams = JSON.parse(monParamsStr);
      params = {
        ...params,
        ...monitorParams
      };
    }
    if (!(0, _lodash.isEmpty)(params)) {
      return {
        str: JSON.stringify(params),
        params
      };
    } else {
      return {
        str: '',
        params
      };
    }
  } catch (e) {
    // ignore
  }
  return {
    str: paramsString,
    params
  };
};
exports.mixParamsWithGlobalParams = mixParamsWithGlobalParams;