"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchEsqlQuery = fetchEsqlQuery;
exports.generateLink = generateLink;
exports.getSourceFields = exports.getEsqlQuery = void 0;
var _lodash = require("lodash");
var _common = require("@kbn/triggers-actions-ui-plugin/common");
var _alertsAsDataUtils = require("@kbn/alerts-as-data-utils");
var _server = require("@kbn/task-manager-plugin/server");
var _common2 = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchEsqlQuery({
  ruleId,
  alertLimit,
  params,
  services,
  spacePrefix,
  dateStart,
  dateEnd
}) {
  const {
    logger,
    scopedClusterClient,
    share,
    ruleResultService
  } = services;
  const discoverLocator = share.url.locators.get('DISCOVER_APP_LOCATOR');
  const esClient = scopedClusterClient.asCurrentUser;
  const query = getEsqlQuery(params, alertLimit, dateStart, dateEnd);
  logger.debug(() => `ES|QL query rule (${ruleId}) query: ${JSON.stringify(query)}`);
  let response;
  try {
    response = await esClient.transport.request({
      method: 'POST',
      path: '/_query',
      body: query
    });
  } catch (e) {
    var _e$message;
    if ((_e$message = e.message) !== null && _e$message !== void 0 && _e$message.includes('verification_exception')) {
      throw (0, _server.createTaskRunError)(e, _server.TaskErrorSource.USER);
    }
    throw e;
  }
  const sourceFields = getSourceFields(response);
  const isGroupAgg = (0, _common.isPerRowAggregation)(params.groupBy);
  const {
    results,
    duplicateAlertIds
  } = (0, _common2.getEsqlQueryHits)(response, params.esqlQuery.esql, isGroupAgg);
  if (ruleResultService && duplicateAlertIds && duplicateAlertIds.size > 0) {
    const warning = `The query returned multiple rows with the same alert ID. There are duplicate results for alert IDs: ${Array.from(duplicateAlertIds).join('; ')}`;
    ruleResultService.addLastRunWarning(warning);
    ruleResultService.setLastRunOutcomeMessage(warning);
  }
  const link = generateLink(params, discoverLocator, dateStart, dateEnd, spacePrefix);
  return {
    link,
    parsedResults: (0, _common.parseAggregationResults)({
      ...results,
      resultLimit: alertLimit,
      sourceFieldsParams: sourceFields,
      generateSourceFieldsFromHits: true
    }),
    index: null
  };
}
const getEsqlQuery = (params, alertLimit, dateStart, dateEnd) => {
  const rangeFilter = [{
    range: {
      [params.timeField]: {
        lte: dateEnd,
        gt: dateStart,
        format: 'strict_date_optional_time'
      }
    }
  }];
  const query = {
    query: alertLimit ? `${params.esqlQuery.esql} | limit ${alertLimit}` : params.esqlQuery.esql,
    filter: {
      bool: {
        filter: rangeFilter
      }
    }
  };
  return query;
};
exports.getEsqlQuery = getEsqlQuery;
const getSourceFields = results => {
  const resultFields = results.columns.map(c => ({
    label: c.name,
    searchPath: c.name
  }));
  const alertFields = Object.keys(_alertsAsDataUtils.alertFieldMap);
  const ecsFields = Object.keys(_alertsAsDataUtils.ecsFieldMap)
  // exclude the alert fields that we don't want to override
  .filter(key => !alertFields.includes(key)).map(key => ({
    label: key,
    searchPath: key
  }));
  return (0, _lodash.intersectionBy)(resultFields, ecsFields, 'label');
};
exports.getSourceFields = getSourceFields;
function generateLink(params, discoverLocator, dateStart, dateEnd, spacePrefix) {
  const redirectUrlParams = {
    timeRange: {
      from: dateStart,
      to: dateEnd
    },
    query: params.esqlQuery,
    isAlertResults: true
  };

  // use `lzCompress` flag for making the link readable during debugging/testing
  // const redirectUrl = discoverLocator!.getRedirectUrl(redirectUrlParams, { lzCompress: false });
  const redirectUrl = discoverLocator.getRedirectUrl(redirectUrlParams, {
    spaceId: spacePrefix
  });
  return redirectUrl;
}