"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SpacesClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _common = require("../../common");
var _space_solution_disabled_features = require("../lib/utils/space_solution_disabled_features");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SUPPORTED_GET_SPACE_PURPOSES = ['any', 'copySavedObjectsIntoSpace', 'findSavedObjects', 'shareSavedObjectsIntoSpace'];
const DEFAULT_PURPOSE = 'any';
const LEGACY_URL_ALIAS_TYPE = 'legacy-url-alias';

/**
 * Client interface for interacting with spaces.
 */

/**
 * Client for interacting with spaces.
 */
class SpacesClient {
  constructor(debugLogger, config, repository, nonGlobalTypeNames, buildFlavour, features) {
    (0, _defineProperty2.default)(this, "isServerless", void 0);
    /**
     * A map of deprecated feature IDs to the feature IDs that replace them used to transform the disabled features
     * of a space to make sure they only reference non-deprecated features.
     */
    (0, _defineProperty2.default)(this, "deprecatedFeaturesReferences", void 0);
    (0, _defineProperty2.default)(this, "transformSavedObjectToSpace", savedObject => {
      var _savedObject$attribut, _savedObject$attribut2, _savedObject$attribut3;
      // Solution isn't supported in the serverless offering.
      const solution = !this.isServerless ? savedObject.attributes.solution : undefined;
      return {
        id: savedObject.id,
        name: (_savedObject$attribut = savedObject.attributes.name) !== null && _savedObject$attribut !== void 0 ? _savedObject$attribut : '',
        description: savedObject.attributes.description,
        color: savedObject.attributes.color,
        initials: savedObject.attributes.initials,
        imageUrl: savedObject.attributes.imageUrl,
        disabledFeatures: (0, _space_solution_disabled_features.withSpaceSolutionDisabledFeatures)(this.features.getKibanaFeatures(), (_savedObject$attribut2 = (_savedObject$attribut3 = savedObject.attributes.disabledFeatures) === null || _savedObject$attribut3 === void 0 ? void 0 : _savedObject$attribut3.flatMap(featureId => {
          var _this$deprecatedFeatu;
          return Array.from((_this$deprecatedFeatu = this.deprecatedFeaturesReferences.get(featureId)) !== null && _this$deprecatedFeatu !== void 0 ? _this$deprecatedFeatu : [featureId]);
        })) !== null && _savedObject$attribut2 !== void 0 ? _savedObject$attribut2 : [], solution),
        _reserved: savedObject.attributes._reserved,
        ...(solution ? {
          solution
        } : {})
      };
    });
    (0, _defineProperty2.default)(this, "generateSpaceAttributes", space => {
      return {
        name: space.name,
        description: space.description,
        color: space.color,
        initials: space.initials,
        imageUrl: space.imageUrl,
        disabledFeatures: space.disabledFeatures,
        ...(!this.isServerless && space.solution ? {
          solution: space.solution
        } : {})
      };
    });
    this.debugLogger = debugLogger;
    this.config = config;
    this.repository = repository;
    this.nonGlobalTypeNames = nonGlobalTypeNames;
    this.buildFlavour = buildFlavour;
    this.features = features;
    this.isServerless = this.buildFlavour === 'serverless';
    this.deprecatedFeaturesReferences = this.collectDeprecatedFeaturesReferences(features.getKibanaFeatures());
  }
  async getAll(options = {}) {
    const {
      purpose = DEFAULT_PURPOSE
    } = options;
    if (!SUPPORTED_GET_SPACE_PURPOSES.includes(purpose)) {
      throw _boom.default.badRequest(`unsupported space purpose: ${purpose}`);
    }
    this.debugLogger(`SpacesClient.getAll(). querying all spaces`);
    const {
      saved_objects: savedObjects
    } = await this.repository.find({
      type: 'space',
      page: 1,
      perPage: this.config.maxSpaces,
      sortField: 'name.keyword'
    });
    this.debugLogger(`SpacesClient.getAll(). Found ${savedObjects.length} spaces.`);
    return savedObjects.map(this.transformSavedObjectToSpace);
  }
  async get(id) {
    const savedObject = await this.repository.get('space', id);
    return this.transformSavedObjectToSpace(savedObject);
  }
  async create(space) {
    const {
      total
    } = await this.repository.find({
      type: 'space',
      page: 1,
      perPage: 0
    });
    if (total >= this.config.maxSpaces) {
      throw _boom.default.badRequest('Unable to create Space, this exceeds the maximum number of spaces set by the xpack.spaces.maxSpaces setting');
    }
    if (space.disabledFeatures.length > 0 && !this.config.allowFeatureVisibility) {
      throw _boom.default.badRequest('Unable to create Space, the disabledFeatures array must be empty when xpack.spaces.allowFeatureVisibility setting is disabled');
    }
    if (Boolean(space.solution) && !this.config.allowSolutionVisibility) {
      throw _boom.default.badRequest('Unable to create Space, the solution property can not be set when xpack.spaces.allowSolutionVisibility setting is disabled');
    }
    if (this.isServerless && Object.hasOwn(space, 'solution')) {
      throw _boom.default.badRequest('Unable to create Space, solution property is forbidden in serverless');
    }
    if (Object.hasOwn(space, 'solution') && !space.solution) {
      throw _boom.default.badRequest('Unable to create Space, solution property cannot be empty');
    }
    this.debugLogger(`SpacesClient.create(), using RBAC. Attempting to create space`);
    const id = space.id;
    const attributes = this.generateSpaceAttributes(space);
    const createdSavedObject = await this.repository.create('space', attributes, {
      id
    });
    this.debugLogger(`SpacesClient.create(), created space object`);
    return this.transformSavedObjectToSpace(createdSavedObject);
  }
  async update(id, space) {
    if (space.disabledFeatures.length > 0 && !this.config.allowFeatureVisibility) {
      throw _boom.default.badRequest('Unable to update Space, the disabledFeatures array must be empty when xpack.spaces.allowFeatureVisibility setting is disabled');
    }
    if (Boolean(space.solution) && !this.config.allowSolutionVisibility) {
      throw _boom.default.badRequest('Unable to update Space, the solution property can not be set when xpack.spaces.allowSolutionVisibility setting is disabled');
    }
    if (this.isServerless && Object.hasOwn(space, 'solution')) {
      throw _boom.default.badRequest('Unable to update Space, solution property is forbidden in serverless');
    }
    if (Object.hasOwn(space, 'solution') && !space.solution) {
      throw _boom.default.badRequest('Unable to update Space, solution property cannot be empty');
    }
    const attributes = this.generateSpaceAttributes(space);
    await this.repository.update('space', id, attributes);
    const updatedSavedObject = await this.repository.get('space', id);
    return this.transformSavedObjectToSpace(updatedSavedObject);
  }
  createSavedObjectFinder(id) {
    return this.repository.createPointInTimeFinder({
      type: this.nonGlobalTypeNames,
      namespaces: [id]
    });
  }
  async delete(id) {
    const existingSavedObject = await this.repository.get('space', id);
    if ((0, _common.isReservedSpace)(this.transformSavedObjectToSpace(existingSavedObject))) {
      throw _boom.default.badRequest(`The ${id} space cannot be deleted because it is reserved.`);
    }
    await this.repository.deleteByNamespace(id);
    await this.repository.delete('space', id);
  }
  async disableLegacyUrlAliases(aliases) {
    const attributes = {
      disabled: true
    };
    const objectsToUpdate = aliases.map(({
      targetSpace,
      targetType,
      sourceId
    }) => {
      const id = `${targetSpace}:${targetType}:${sourceId}`;
      return {
        type: LEGACY_URL_ALIAS_TYPE,
        id,
        attributes
      };
    });
    await this.repository.bulkUpdate(objectsToUpdate);
  }
  /**
   * Collects a map of all deprecated feature IDs and the feature IDs that replace them.
   * @param features A list of all available Kibana features including deprecated ones.
   */
  collectDeprecatedFeaturesReferences(features) {
    const deprecatedFeatureReferences = new Map();
    for (const feature of features) {
      var _feature$deprecated, _feature$privileges, _feature$subFeatures$, _feature$subFeatures;
      if (!feature.deprecated) {
        continue;
      }

      // If the feature is deprecated and replacement features are explicitly defined, use them.
      // Otherwise, use the replacement features defined in the feature privileges.
      const featureReplacedBy = (_feature$deprecated = feature.deprecated) === null || _feature$deprecated === void 0 ? void 0 : _feature$deprecated.replacedBy;
      if (featureReplacedBy) {
        deprecatedFeatureReferences.set(feature.id, new Set(featureReplacedBy));
        continue;
      }

      // Collect all feature privileges including the ones provided by sub-features, if any.
      const allPrivileges = Object.values((_feature$privileges = feature.privileges) !== null && _feature$privileges !== void 0 ? _feature$privileges : {}).concat((_feature$subFeatures$ = (_feature$subFeatures = feature.subFeatures) === null || _feature$subFeatures === void 0 ? void 0 : _feature$subFeatures.flatMap(subFeature => subFeature.privilegeGroups.flatMap(({
        privileges
      }) => privileges))) !== null && _feature$subFeatures$ !== void 0 ? _feature$subFeatures$ : []);

      // Collect all features IDs that are referenced by the deprecated feature privileges.
      const referencedFeaturesIds = new Set();
      for (const privilege of allPrivileges) {
        const replacedBy = privilege.replacedBy ? 'default' in privilege.replacedBy ? privilege.replacedBy.default.concat(privilege.replacedBy.minimal) : privilege.replacedBy : [];
        for (const privilegeReference of replacedBy) {
          referencedFeaturesIds.add(privilegeReference.feature);
        }
      }
      deprecatedFeatureReferences.set(feature.id, referencedFeaturesIds);
    }
    return deprecatedFeatureReferences;
  }
}
exports.SpacesClient = SpacesClient;