"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultSummaryTransformManager = void 0;
var _errors = require("../errors");
var _retry = require("../utils/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultSummaryTransformManager {
  constructor(generator, scopedClusterClient, logger, abortController = new AbortController()) {
    this.generator = generator;
    this.scopedClusterClient = scopedClusterClient;
    this.logger = logger;
    this.abortController = abortController;
  }
  async install(slo) {
    const transformParams = await this.generator.generate(slo);
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.putTransform(transformParams, {
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      var _err$meta, _err$meta$body, _err$meta$body$error;
      this.logger.debug(`Cannot create summary transform for SLO [${slo.id}]. ${err}`);
      if (((_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) === 'security_exception') {
        throw new _errors.SecurityException(err.meta.body.error.reason);
      }
      throw err;
    }
    return transformParams.transform_id;
  }
  async inspect(slo) {
    return this.generator.generate(slo);
  }
  async preview(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.previewTransform({
        transform_id: transformId
      }, {
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.debug(`Cannot preview SLO summary transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async start(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.startTransform({
        transform_id: transformId
      }, {
        ignore: [409],
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.debug(`Cannot start SLO summary transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async stop(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.stopTransform({
        transform_id: transformId,
        wait_for_completion: true,
        force: true
      }, {
        ignore: [404],
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.debug(`Cannot stop SLO summary transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async uninstall(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.deleteTransform({
        transform_id: transformId,
        force: true
      }, {
        ignore: [404],
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.debug(`Cannot delete SLO summary transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async getVersion(transformId) {
    try {
      var _response$transforms$, _response$transforms$2;
      const response = await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.getTransform({
        transform_id: transformId
      }, {
        ignore: [404],
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
      return response === null || response === void 0 ? void 0 : (_response$transforms$ = response.transforms[0]) === null || _response$transforms$ === void 0 ? void 0 : (_response$transforms$2 = _response$transforms$._meta) === null || _response$transforms$2 === void 0 ? void 0 : _response$transforms$2.version;
    } catch (err) {
      this.logger.debug(`Cannot retrieve SLO transform version [${transformId}]. ${err}`);
      throw err;
    }
  }
}
exports.DefaultSummaryTransformManager = DefaultSummaryTransformManager;