"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsCreateRulesRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _resources = require("../../../../../../common/siem_migrations/rules/resources");
var _audit = require("../util/audit");
var _authz = require("../util/authz");
var _with_existing_migration_id = require("../util/with_existing_migration_id");
var _with_license = require("../util/with_license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsCreateRulesRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_RULE_MIGRATION_RULES_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.CreateRuleMigrationRulesRequestBody),
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.CreateRuleMigrationRulesRequestParams)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const {
      migration_id: migrationId
    } = req.params;
    const originalRules = req.body;
    const rulesCount = originalRules.length;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution);
    try {
      const [firstOriginalRule] = originalRules;
      if (!firstOriginalRule) {
        return res.noContent();
      }
      const ctx = await context.resolve(['securitySolution']);
      const ruleMigrationsClient = ctx.securitySolution.getSiemRuleMigrationsClient();
      await siemMigrationAuditLogger.logAddRules({
        migrationId,
        count: rulesCount
      });
      const ruleMigrations = originalRules.map(originalRule => ({
        migration_id: migrationId,
        original_rule: originalRule
      }));
      await ruleMigrationsClient.data.rules.create(ruleMigrations);

      // Create identified resource documents without content to keep track of them
      const resourceIdentifier = new _resources.ResourceIdentifier(firstOriginalRule.vendor);
      const resources = resourceIdentifier.fromOriginalRules(originalRules).map(resource => ({
        ...resource,
        migration_id: migrationId
      }));
      if (resources.length > 0) {
        await ruleMigrationsClient.data.resources.create(resources);
      }
      return res.ok();
    } catch (error) {
      logger.error(error);
      await siemMigrationAuditLogger.logAddRules({
        migrationId,
        count: rulesCount,
        error
      });
      return res.badRequest({
        body: error.message
      });
    }
  })));
};
exports.registerSiemRuleMigrationsCreateRulesRoute = registerSiemRuleMigrationsCreateRulesRoute;