"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractIntegrations = extractIntegrations;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function extractIntegrations(packages, packagePolicies) {
  const result = [];
  const enabledIntegrationsSet = extractEnabledIntegrations(packagePolicies);
  for (const fleetPackage of packages) {
    var _fleetPackage$savedOb, _fleetPackage$policy_;
    const packageName = fleetPackage.name;
    const packageTitle = fleetPackage.title;
    const isPackageInstalled = fleetPackage.status === 'installed';
    // Actual `installed_version` is buried in SO, root `version` is latest package version available
    const installedPackageVersion = (_fleetPackage$savedOb = fleetPackage.savedObject) === null || _fleetPackage$savedOb === void 0 ? void 0 : _fleetPackage$savedOb.attributes.install_version;
    // Policy templates correspond to package's integrations.
    const packagePolicyTemplates = (_fleetPackage$policy_ = fleetPackage.policy_templates) !== null && _fleetPackage$policy_ !== void 0 ? _fleetPackage$policy_ : [];
    for (const policyTemplate of packagePolicyTemplates) {
      const integrationName = policyTemplate.name;
      if (integrationName !== packageName) {
        const integrationId = getIntegrationId(packageName, integrationName);
        const integrationTitle = `${packageTitle} ${(0, _lodash.capitalize)(policyTemplate.title)}`;
        const integration = {
          package_name: packageName,
          package_title: packageTitle,
          latest_package_version: fleetPackage.version,
          installed_package_version: installedPackageVersion,
          integration_name: integrationName,
          integration_title: integrationTitle,
          is_installed: isPackageInstalled,
          // All integrations installed as a part of the package
          is_enabled: enabledIntegrationsSet.has(integrationId)
        };
        result.push(integration);
      }
    }

    // There are two edge cases here
    //
    // - (1) Some prebuilt rules don't use integration name when there is just
    //   one integration per package, e.g. "Web Application Suspicious Activity:
    //   Unauthorized Method" refers "apm" package name while apm package has
    //   "apmserver" integration
    //
    // - (2) Some packages don't have policy templates at all,
    //   e.g. "Lateral Movement Detection"
    if (packagePolicyTemplates.length <= 1) {
      result.push({
        package_name: packageName,
        package_title: packageTitle,
        latest_package_version: fleetPackage.version,
        installed_package_version: installedPackageVersion,
        is_installed: isPackageInstalled,
        is_enabled: enabledIntegrationsSet.has(getIntegrationId(packageName, ''))
      });
    }
  }
  return result;
}
function extractEnabledIntegrations(packagePolicies) {
  const enabledIntegrations = new Set();
  for (const packagePolicy of packagePolicies) {
    for (const input of packagePolicy.inputs) {
      if (input.enabled) {
        var _packagePolicy$packag, _packagePolicy$packag2, _ref, _input$policy_templat;
        const packageName = (_packagePolicy$packag = (_packagePolicy$packag2 = packagePolicy.package) === null || _packagePolicy$packag2 === void 0 ? void 0 : _packagePolicy$packag2.name.trim()) !== null && _packagePolicy$packag !== void 0 ? _packagePolicy$packag : ''; // e.g. 'cloudtrail'
        const integrationName = ((_ref = (_input$policy_templat = input.policy_template) !== null && _input$policy_templat !== void 0 ? _input$policy_templat : input.type) !== null && _ref !== void 0 ? _ref : '').trim(); // e.g. 'cloudtrail'
        const enabledIntegrationKey = `${packageName}${integrationName}`;
        enabledIntegrations.add(enabledIntegrationKey);
      }
    }

    // Base package may not have policy template, so pull directly from `policy.package` if so
    if (packagePolicy.package) {
      const packageName = packagePolicy.package.name.trim();
      enabledIntegrations.add(packageName);
    }
  }
  return enabledIntegrations;
}
function getIntegrationId(packageName, integrationName) {
  return `${packageName}${integrationName}`;
}