"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.actionListHandler = void 0;
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _constants = require("../../../../common/endpoint/constants");
var _services = require("../../services");
var _error_handler = require("../error_handler");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatRequestParams = value => typeof value === 'string' ? [value] : value;
const formatStatusValues = value => typeof value === 'string' ? [value] : value;
const actionListHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('endpoint_action_list');
  return async (context, req, res) => {
    const {
      query: {
        agentIds: elasticAgentIds,
        agentTypes: _agentTypes,
        page,
        pageSize,
        startDate,
        endDate,
        userIds,
        commands,
        statuses,
        withOutputs,
        types
      }
    } = req;
    const esClient = (await context.core).elasticsearch.client.asInternalUser;
    try {
      const indexExists = await (0, _utils.doesLogsEndpointActionsIndexExist)({
        esClient,
        logger,
        indexName: _constants.ENDPOINT_ACTIONS_INDEX
      });
      if (!indexExists) {
        return res.notFound({
          body: 'index_not_found_exception'
        });
      }

      // verify feature flag for sentinel_one `aaentType`
      const agentTypes = formatRequestParams(_agentTypes);
      if (!endpointContext.experimentalFeatures.responseActionsSentinelOneV1Enabled && agentTypes !== null && agentTypes !== void 0 && agentTypes.includes('sentinel_one')) {
        return (0, _error_handler.errorHandler)(logger, res, new _custom_http_request_error.CustomHttpRequestError('[request body.agentTypes]: sentinel_one is disabled', 400));
      }
      const requestParams = {
        agentTypes,
        withOutputs: formatRequestParams(withOutputs),
        types: formatRequestParams(types),
        commands: formatRequestParams(commands),
        esClient,
        elasticAgentIds: formatRequestParams(elasticAgentIds),
        metadataService: endpointContext.service.getEndpointMetadataService(),
        page,
        pageSize,
        startDate,
        endDate,
        userIds: formatRequestParams(userIds),
        logger
      };
      // wrapper method to branch logic for
      // normal paged search via page, size
      // vs full search for status filters
      const getActionsLog = () => {
        if (statuses !== null && statuses !== void 0 && statuses.length) {
          return (0, _services.getActionListByStatus)({
            ...requestParams,
            statuses: formatStatusValues(statuses)
          });
        }
        return (0, _services.getActionList)(requestParams);
      };
      const body = await getActionsLog();
      return res.ok({
        body
      });
    } catch (error) {
      return (0, _error_handler.errorHandler)(logger, res, error);
    }
  };
};
exports.actionListHandler = actionListHandler;