"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = exports.postEnableSpaceAwarenessHandler = exports.getCheckPermissionsHandler = exports.getAgentPoliciesSpacesHandler = exports.generateServiceTokenHandler = void 0;
var _experimental_features = require("../../../common/experimental_features");
var _constants = require("../../constants");
var _constants2 = require("../../../common/constants");
var _services = require("../../services");
var _errors = require("../../errors");
var _types = require("../../types");
var _enable_space_awareness = require("../../services/spaces/enable_space_awareness");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getCheckPermissionsHandler = async (context, request, response) => {
  var _appContextService$ge, _appContextService$ge2;
  const missingSecurityBody = {
    success: false,
    error: 'MISSING_SECURITY'
  };
  const isServerless = (_appContextService$ge = _services.appContextService.getCloud()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.isServerlessEnabled;
  const isSubfeaturePrivilegesEnabled = (_appContextService$ge2 = _services.appContextService.getExperimentalFeatures().subfeaturePrivileges) !== null && _appContextService$ge2 !== void 0 ? _appContextService$ge2 : false;
  if (!_services.appContextService.getSecurityLicense().isEnabled()) {
    return response.ok({
      body: missingSecurityBody
    });
  } else if (isSubfeaturePrivilegesEnabled) {
    const fleetContext = await context.fleet;
    if (!fleetContext.authz.fleet.all && !fleetContext.authz.fleet.readAgents && !fleetContext.authz.fleet.readAgentPolicies && !fleetContext.authz.fleet.readSettings) {
      return response.ok({
        body: {
          success: false,
          error: 'MISSING_PRIVILEGES'
        }
      });
    }
    // check the manage_service_account cluster privilege only on stateful
    else if (request.query.fleetServerSetup && !isServerless) {
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const {
        has_all_requested: hasAllPrivileges
      } = await esClient.security.hasPrivileges({
        body: {
          cluster: ['manage_service_account']
        }
      });
      if (!hasAllPrivileges) {
        return response.ok({
          body: {
            success: false,
            error: 'MISSING_FLEET_SERVER_SETUP_PRIVILEGES'
          }
        });
      }
    }
    return response.ok({
      body: {
        success: true
      }
    });
  } else {
    const fleetContext = await context.fleet;
    if (!fleetContext.authz.fleet.all) {
      return response.ok({
        body: {
          success: false,
          error: 'MISSING_PRIVILEGES'
        }
      });
    }
    // check the manage_service_account cluster privilege only on stateful
    else if (request.query.fleetServerSetup && !isServerless) {
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const {
        has_all_requested: hasAllPrivileges
      } = await esClient.security.hasPrivileges({
        body: {
          cluster: ['manage_service_account']
        }
      });
      if (!hasAllPrivileges) {
        return response.ok({
          body: {
            success: false,
            error: 'MISSING_FLEET_SERVER_SETUP_PRIVILEGES'
          }
        });
      }
    }
    return response.ok({
      body: {
        success: true
      }
    });
  }
};
exports.getCheckPermissionsHandler = getCheckPermissionsHandler;
const postEnableSpaceAwarenessHandler = async (context, request, response) => {
  await (0, _enable_space_awareness.enableSpaceAwarenessMigration)();
  return response.ok({
    body: {}
  });
};
exports.postEnableSpaceAwarenessHandler = postEnableSpaceAwarenessHandler;
const generateServiceTokenHandler = async (context, request, response) => {
  // Generate the fleet server service token as the current user as the internal user do not have the correct permissions
  const esClient = (await context.core).elasticsearch.client.asCurrentUser;
  const serviceAccount = request.body.remote ? 'fleet-server-remote' : 'fleet-server';
  _services.appContextService.getLogger().debug(`Creating service token for account elastic/${serviceAccount}`);
  try {
    const tokenResponse = await esClient.transport.request({
      method: 'POST',
      path: `_security/service/elastic/${serviceAccount}/credential/token/token-${Date.now()}`
    });
    if (tokenResponse.created && tokenResponse.token) {
      const body = tokenResponse.token;
      return response.ok({
        body
      });
    } else {
      const error = new _errors.GenerateServiceTokenError('Unable to generate service token');
      throw error;
    }
  } catch (e) {
    const error = new _errors.GenerateServiceTokenError(e);
    throw error;
  }
};
exports.generateServiceTokenHandler = generateServiceTokenHandler;
const getAgentPoliciesSpacesHandler = async (context, request, response) => {
  const spaces = await (await context.fleet).getAllSpaces();
  const security = _services.appContextService.getSecurity();
  const spaceIds = spaces.map(({
    id
  }) => id);
  const res = await security.authz.checkPrivilegesWithRequest(request).atSpaces(spaceIds, {
    kibana: [security.authz.actions.api.get(`fleet-agent-policies-all`)]
  });
  const authorizedSpaces = spaces.filter(space => {
    var _res$privileges$kiban, _res$privileges$kiban2;
    return (_res$privileges$kiban = (_res$privileges$kiban2 = res.privileges.kibana.find(privilege => privilege.resource === space.id)) === null || _res$privileges$kiban2 === void 0 ? void 0 : _res$privileges$kiban2.authorized) !== null && _res$privileges$kiban !== void 0 ? _res$privileges$kiban : false;
  });
  return response.ok({
    body: {
      items: authorizedSpaces
    }
  });
};
exports.getAgentPoliciesSpacesHandler = getAgentPoliciesSpacesHandler;
const serviceTokenBodyValidation = (data, validationResult) => {
  const {
    ok
  } = validationResult;
  if (!data) {
    return ok({
      remote: false
    });
  }
  const {
    remote
  } = data;
  return ok({
    remote
  });
};
const registerRoutes = (router, config) => {
  const experimentalFeatures = (0, _experimental_features.parseExperimentalConfigValue)(config.enableExperimental);
  if (experimentalFeatures.useSpaceAwareness) {
    router.versioned.post({
      path: '/internal/fleet/enable_space_awareness',
      access: 'internal',
      fleetAuthz: {
        fleet: {
          all: true
        }
      }
    }).addVersion({
      version: _constants2.API_VERSIONS.internal.v1,
      validate: {}
    }, postEnableSpaceAwarenessHandler);
  }
  router.versioned.get({
    path: _constants.APP_API_ROUTES.CHECK_PERMISSIONS_PATTERN,
    summary: `Check permissions`,
    options: {
      tags: ['oas-tag:Fleet internals']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: _types.CheckPermissionsRequestSchema
    }
  }, getCheckPermissionsHandler);
  router.versioned.get({
    path: _constants.APP_API_ROUTES.AGENT_POLICIES_SPACES,
    access: 'internal',
    fleetAuthz: {
      fleet: {
        readAgentPolicies: true
      }
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.internal.v1,
    validate: {}
  }, getAgentPoliciesSpacesHandler);
  router.versioned.post({
    path: _constants.APP_API_ROUTES.GENERATE_SERVICE_TOKEN_PATTERN,
    fleetAuthz: {
      fleet: {
        allAgents: true
      }
    },
    summary: `Create a service token`,
    options: {
      tags: ['oas-tag:Fleet service tokens']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: serviceTokenBodyValidation
      }
    }
  }, generateServiceTokenHandler);
  router.versioned.post({
    path: _constants.APP_API_ROUTES.GENERATE_SERVICE_TOKEN_PATTERN_DEPRECATED,
    fleetAuthz: {
      fleet: {
        allAgents: true
      }
    },
    description: `Create a service token`
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {}
  }, generateServiceTokenHandler);
};
exports.registerRoutes = registerRoutes;