"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOrUpdateKnowledgeBaseIndexAssets = createOrUpdateKnowledgeBaseIndexAssets;
var _server = require("@kbn/alerting-plugin/server");
var _kb_component_template = require("./templates/kb_component_template");
var _ = require("..");
var _get_inference_id_from_write_index = require("../knowledge_base_service/get_inference_id_from_write_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function createOrUpdateKnowledgeBaseIndexAssets({
  logger,
  core,
  inferenceId: componentTemplateInferenceId
}) {
  try {
    logger.debug('Setting up knowledge base index assets');
    const [coreStart] = await core.getStartServices();
    const esClient = coreStart.elasticsearch.client;
    const {
      asInternalUser
    } = esClient;

    // Knowledge base: component template
    await asInternalUser.cluster.putComponentTemplate({
      create: false,
      name: _.resourceNames.componentTemplate.kb,
      template: (0, _kb_component_template.getComponentTemplate)(componentTemplateInferenceId)
    });
    logger.debug(`Knowledge base component template updated with inference_id: ${componentTemplateInferenceId}`);

    // Knowledge base: index template
    await asInternalUser.indices.putIndexTemplate({
      name: _.resourceNames.indexTemplate.kb,
      composed_of: [_.resourceNames.componentTemplate.kb],
      create: false,
      index_patterns: [_.resourceNames.indexPatterns.kb],
      template: {
        settings: {
          number_of_shards: 1,
          auto_expand_replicas: '0-1',
          hidden: true
        }
      }
    });
    const writeIndexInferenceId = await (0, _get_inference_id_from_write_index.getInferenceIdFromWriteIndex)(esClient).catch(() => undefined);
    logger.debug(`Current write index inference id: ${writeIndexInferenceId}`);

    // Knowledge base: write index
    // `createConcreteWriteIndex` will create the write index, or update the index mappings if the index already exists
    // only invoke `createConcreteWriteIndex` if the write index does not exist or the inferenceId in the component template is the same as the one in the write index
    if (!writeIndexInferenceId || writeIndexInferenceId === componentTemplateInferenceId) {
      const kbAliasName = _.resourceNames.writeIndexAlias.kb;
      await (0, _server.createConcreteWriteIndex)({
        esClient: asInternalUser,
        logger,
        totalFieldsLimit: 10000,
        indexPatterns: {
          alias: kbAliasName,
          pattern: `${kbAliasName}*`,
          basePattern: `${kbAliasName}*`,
          name: _.resourceNames.concreteWriteIndexName.kb,
          template: _.resourceNames.indexTemplate.kb
        },
        dataStreamAdapter: (0, _server.getDataStreamAdapter)({
          useDataStreamForAlerts: false
        })
      });
    }
    logger.info('Successfully set up knowledge base index assets');
  } catch (error) {
    logger.error(`Failed setting up knowledge base index assets: ${error.message}`);
    logger.debug(error);
  }
}