"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initMetricExplorerRoute = void 0;
var _ioTsUtils = require("@kbn/io-ts-utils");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _constants = require("../../../common/constants");
var _metrics_explorer = require("../../../common/http_api/metrics_explorer");
var _convert_request_to_metrics_api_options = require("./lib/convert_request_to_metrics_api_options");
var _create_search_client = require("../../lib/create_search_client");
var _find_interval_for_metrics = require("./lib/find_interval_for_metrics");
var _metrics = require("../../lib/metrics");
var _query_total_groupings = require("./lib/query_total_groupings");
var _transform_series = require("./lib/transform_series");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const initMetricExplorerRoute = framework => {
  const validateBody = (0, _ioTsUtils.createRouteValidationFunction)(_metrics_explorer.metricsExplorerRequestBodyRT);
  framework.registerRoute({
    method: 'post',
    path: '/api/infra/metrics_explorer',
    validate: {
      body: validateBody
    }
  }, async (requestContext, request, response) => {
    const options = request.body;
    try {
      if (options.metrics.length > _constants.METRICS_EXPLORER_API_MAX_METRICS) {
        throw _boom.default.badRequest(`'metrics' size is greater than maximum of ${_constants.METRICS_EXPLORER_API_MAX_METRICS} allowed.`);
      }
      const client = (0, _create_search_client.createSearchClient)(requestContext, framework);
      const interval = await (0, _find_interval_for_metrics.findIntervalForMetrics)(client, options);
      const optionsWithInterval = options.forceInterval ? options : {
        ...options,
        timerange: {
          ...options.timerange,
          interval: interval ? `>=${interval}s` : options.timerange.interval
        }
      };
      const metricsApiOptions = (0, _convert_request_to_metrics_api_options.convertRequestToMetricsAPIOptions)(optionsWithInterval);
      const metricsApiResponse = await (0, _metrics.query)(client, metricsApiOptions);
      const totalGroupings = await (0, _query_total_groupings.queryTotalGroupings)(client, metricsApiOptions);
      const hasGroupBy = Array.isArray(metricsApiOptions.groupBy) && metricsApiOptions.groupBy.length > 0;
      const pageInfo = {
        total: totalGroupings,
        afterKey: null
      };
      if (metricsApiResponse.info.afterKey) {
        pageInfo.afterKey = metricsApiResponse.info.afterKey;
      }

      // If we have a groupBy but there are ZERO groupings returned then we need to
      // return an empty array. Otherwise we transform the series to match the current schema.
      const series = hasGroupBy && totalGroupings === 0 ? [] : metricsApiResponse.series.map((0, _transform_series.transformSeries)(hasGroupBy));
      return response.ok({
        body: _metrics_explorer.metricsExplorerResponseRT.encode({
          series,
          pageInfo
        })
      });
    } catch (err) {
      var _err$statusCode, _err$message;
      if (_boom.default.isBoom(err)) {
        return response.customError({
          statusCode: err.output.statusCode,
          body: {
            message: err.output.payload.message
          }
        });
      }
      return response.customError({
        statusCode: (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500,
        body: {
          message: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : 'An unexpected error occurred'
        }
      });
    }
  });
};
exports.initMetricExplorerRoute = initMetricExplorerRoute;