"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.applyPaletteParams = applyPaletteParams;
exports.findMinMaxByColumnId = void 0;
exports.getDataMinMax = getDataMinMax;
exports.getFormattedTable = exports.getFormattedRow = void 0;
exports.getNumericValue = getNumericValue;
exports.getSortPredicate = void 0;
exports.remapStopsByNewInterval = remapStopsByNewInterval;
var _coloring = require("@kbn/coloring");
var _transposeUtils = require("@kbn/transpose-utils");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getDataMinMax(rangeType, dataBounds) {
  const dataMin = rangeType === 'number' ? dataBounds.min : _coloring.DEFAULT_MIN_STOP;
  const dataMax = rangeType === 'number' ? dataBounds.max : _coloring.DEFAULT_MAX_STOP;
  return {
    min: dataMin,
    max: dataMax
  };
}
// Utility to remap color stops within new domain
function remapStopsByNewInterval(controlStops, {
  newInterval,
  oldInterval,
  newMin,
  oldMin
}) {
  return (controlStops || []).map(({
    color,
    stop
  }) => {
    return {
      color,
      stop: newMin + (stop - oldMin) * newInterval / oldInterval
    };
  });
}

/**
 * Some name conventions here:
 * * `displayStops` => It's an additional transformation of `stops` into a [0, N] domain for the EUIPaletteDisplay component.
 * * `stops` => final steps used to table coloring. It is a rightShift of the colorStops
 * * `colorStops` => user's color stop inputs.  Used to compute range min.
 *
 * When the user inputs the colorStops, they are designed to be the initial part of the color segment,
 * so the next stops indicate where the previous stop ends.
 * Both table coloring logic and EuiPaletteDisplay format implementation works differently than our current `colorStops`,
 * by having the stop values at the end of each color segment rather than at the beginning: `stops` values are computed by a rightShift of `colorStops`.
 * EuiPaletteDisplay has an additional requirement as it is always mapped against a domain [0, N]: from `stops` the `displayStops` are computed with
 * some continuity enrichment and a remap against a [0, 100] domain to make the palette component work ok.
 *
 * These naming conventions would be useful to track the code flow in this feature as multiple transformations are happening
 * for a single change.
 */

function applyPaletteParams(palettes, activePalette, dataBounds) {
  var _activePalette$name, _activePalette$params, _activePalette$params2;
  // make a copy of it as they have to be manipulated later on
  let displayStops = (0, _coloring.getPaletteStops)(palettes, (activePalette === null || activePalette === void 0 ? void 0 : activePalette.params) || {}, {
    dataBounds,
    defaultPaletteName: (_activePalette$name = activePalette === null || activePalette === void 0 ? void 0 : activePalette.name) !== null && _activePalette$name !== void 0 ? _activePalette$name : _constants.defaultPaletteParams.name
  });
  if (activePalette !== null && activePalette !== void 0 && (_activePalette$params = activePalette.params) !== null && _activePalette$params !== void 0 && _activePalette$params.reverse && (activePalette === null || activePalette === void 0 ? void 0 : (_activePalette$params2 = activePalette.params) === null || _activePalette$params2 === void 0 ? void 0 : _activePalette$params2.name) !== _coloring.CUSTOM_PALETTE) {
    displayStops = (0, _coloring.reversePalette)(displayStops);
  }
  return displayStops;
}
function getNumericValue(rowValue) {
  if (rowValue == null || Array.isArray(rowValue)) {
    return;
  }
  return rowValue;
}
const findMinMaxByColumnId = (columnIds, table) => {
  const minMax = {};
  if (table != null) {
    for (const columnId of columnIds) {
      const originalId = (0, _transposeUtils.getOriginalId)(columnId);
      minMax[originalId] = minMax[originalId] || {
        max: -Infinity,
        min: Infinity
      };
      table.rows.forEach(row => {
        const rowValue = row[columnId];
        const numericValue = getNumericValue(rowValue);
        if (numericValue != null) {
          if (minMax[originalId].min > numericValue) {
            minMax[originalId].min = numericValue;
          }
          if (minMax[originalId].max < numericValue) {
            minMax[originalId].max = numericValue;
          }
        }
      });
      // what happens when there's no data in the table? Fallback to a percent range
      if (minMax[originalId].max === -Infinity) {
        minMax[originalId] = {
          max: 100,
          min: 0,
          fallback: true
        };
      }
    }
  }
  return minMax;
};
exports.findMinMaxByColumnId = findMinMaxByColumnId;
const getSortPredicate = column => {
  var _column$meta, _column$meta$sourcePa;
  const params = (_column$meta = column.meta) === null || _column$meta === void 0 ? void 0 : (_column$meta$sourcePa = _column$meta.sourceParams) === null || _column$meta$sourcePa === void 0 ? void 0 : _column$meta$sourcePa.params;
  const sort = params === null || params === void 0 ? void 0 : params.orderBy;
  if (params !== null && params !== void 0 && params.otherBucket || !sort) return 'dataIndex';
  // metric sorting
  if (sort && sort !== '_key') {
    if ((params === null || params === void 0 ? void 0 : params.order) === 'desc') {
      return 'numDesc';
    } else {
      return 'numAsc';
    }
    // alphabetical sorting
  } else {
    if ((params === null || params === void 0 ? void 0 : params.order) === 'desc') {
      return 'alphaDesc';
    } else {
      return 'alphaAsc';
    }
  }
};
exports.getSortPredicate = getSortPredicate;
const isPrimitive = value => value != null && typeof value !== 'object';
const getFormattedRow = (row, columns, columnsFormatters) => columns.reduce((formattedInfo, {
  id
}) => {
  const record = formattedInfo.row[id];
  if (record != null &&
  // pre-format non-primitive values
  !isPrimitive(record)) {
    return {
      row: {
        ...formattedInfo.row,
        [id]: columnsFormatters[id].convert(record)
      },
      formattedColumns: {
        ...formattedInfo.formattedColumns,
        [id]: true
      }
    };
  }
  return formattedInfo;
}, {
  row,
  formattedColumns: {}
});
exports.getFormattedRow = getFormattedRow;
const getFormattedTable = (table, formatFactory) => {
  const columnsFormatters = table.columns.reduce((formatters, {
    id,
    meta
  }) => {
    return {
      ...formatters,
      [id]: formatFactory(meta.params)
    };
  }, {});
  const formattedTableInfo = {
    rows: [],
    formattedColumns: {}
  };
  for (const row of table.rows) {
    const formattedRowInfo = getFormattedRow(row, table.columns, columnsFormatters);
    formattedTableInfo.rows.push(formattedRowInfo.row);
    formattedTableInfo.formattedColumns = {
      ...formattedTableInfo.formattedColumns,
      ...formattedRowInfo.formattedColumns
    };
  }
  return {
    table: {
      ...table,
      rows: formattedTableInfo.rows
    },
    formattedColumns: formattedTableInfo.formattedColumns
  };
};
exports.getFormattedTable = getFormattedTable;