"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.convertRequests = exports.listFormats = void 0;
const parse_1 = require("./parse");
const python_1 = require("./exporters/python");
const curl_1 = require("./exporters/curl");
const javascript_1 = require("./exporters/javascript");
const EXPORTERS = {
    python: new python_1.PythonExporter(),
    curl: new curl_1.CurlExporter(),
    javascript: new javascript_1.JavaScriptExporter(),
};
/**
 * Return the list of available export formats.
 *
 * @returns An array of strings with the names of the formats that are available
 *   to use in the `convertRequests()` function.
 */
function listFormats() {
    return Object.keys(EXPORTERS);
}
exports.listFormats = listFormats;
/**
 * Convert Elasticsearch requests in Dev Console syntax to other formats.
 *
 * @param source The source Dev Console code, given as a single string. Multiple
 *   requests can be separated with an empty line.
 * @param outputFormat The format to convert to, such as `"python"` or `"javascript"`.
 * @param options Conversion options.
 * @returns When `checkOnly` is set to `true` in `options`, the return value is a
 *   boolean that indicates if the given source code can be converted. A `false`
 *   return indicates that the requested conversion cannot be completed, for
 *   example due to unsupported features in the specified target format. If
 *   `checkOnly` is `false` or not given, the return value is a string with the
 *   converted code.
 */
async function convertRequests(source, outputFormat, options) {
    const requests = await (0, parse_1.parseRequests)(source);
    const exporter = typeof outputFormat == "string" ? EXPORTERS[outputFormat] : outputFormat;
    if (!exporter) {
        throw new Error("Invalid output format");
    }
    /* istanbul ignore next */
    if (options.debug) {
        console.log(JSON.stringify(requests));
    }
    if (options.checkOnly) {
        return await exporter.check(requests);
    }
    return await exporter.convert(requests, options);
}
exports.convertRequests = convertRequests;
