# frozen_string_literal: true

module Actastic
  class Stats

    # Measure the time taken by the given methods. The result is equivalent to this:
    #
    # def my_method
    #   Swiftype::Stats.measure("#{key_prefix}.my_method") do
    #     ...
    #   end
    # end
    #
    # This creates new methods in a Module and prepends the module to the class
    # so that the new methods can call the original implementation with super.
    #
    # klass        - The Class instance to decorate with measured methods.
    # key_prefix   - A String to prepend to the metrics key.
    # method_names - One or more Symbols representing instance methods of klass.
    def self.measure_methods(klass, key_prefix, *method_names)
      measurement_module = Module.new do
        method_names.each do |method_name|
          stats_key = "#{key_prefix}.#{method_name}".freeze
          define_method(method_name) do |*args, **kwargs, &block|
            StatsD.measure(stats_key) do
              super(*args, **kwargs, &block)
            end
          end
        end
      end

      klass.prepend(measurement_module)
    end
  end
end
