# frozen_string_literal: true
module Actastic
  module Validations
    extend ActiveSupport::Concern

    class UniquenessValidator < ActiveModel::EachValidator # :nodoc:
      def initialize(options)
        if options[:conditions] && !options[:conditions].respond_to?(:call)
          raise ArgumentError, "#{options[:conditions]} was passed as :conditions but is not callable. " \
                               'Pass a callable instead: `conditions: -> { where(approved: true) }`'
        end
        super({ :case_sensitive => true }.merge!(options))
        @klass = options[:class]
      end

      def validate_each(record, attribute, value)
        raise unless options.fetch(:case_sensitive, true)

        conditions = options[:conditions]
        scope = [attribute] + Array.wrap(options[:scope])

        relation = conditions ? conditions.call : @klass.relation
        relation = relation.not(:id => record.id) unless record.new_record?
        attributes = scope.each_with_object({}) do |field, out|
          out[field] = record.public_send(field)
        end

        if relation.exists?(**attributes)
          error_options = options.except(:case_sensitive, :scope, :conditions)
          error_options[:value] = value

          record.errors.add(attribute, :taken, **error_options)
        end
      end
    end

    class_methods do
      def validates_uniqueness_of(*attributes)
        validates_with(Actastic::Validations::UniquenessValidator, _merge_attributes(attributes))
      end
    end

    def valid?(context = nil)
      context ||= new_record? ? :create : :update
      super(context)
    end
  end
end
