# frozen_string_literal: true

module Actastic
  class Relation
    include Enumerable
    include Actastic::Concerns::Coercable

    VALID_CONDITION_TYPES = [NilClass, TrueClass, FalseClass, Numeric, String, Symbol, BSON::ObjectId, Range].freeze
    VALID_RANGE_FIELD_TYPES = %w(date keyword long double).freeze
    # Elasticsearch allows for more +collapse+ types, but we're not using those in +TYPE_MAPPINGS+
    VALID_COLLAPSE_FIELD_TYPES = %w(keyword long double).freeze
    VALID_RANGE_TYPES = [Numeric, Time, DateTime, Date, String, Symbol].freeze
    VALID_ORDER_DIRECTIONS = [:asc, :desc].freeze

    DEFAULT_ORDER_DIRECTION = VALID_ORDER_DIRECTIONS.first
    DEFAULT_SIZE_VALUE = 100

    def initialize(klass, values: {}, parent: nil, association_definition: nil, includes_association_definitions: Set[])
      @klass = klass
      @values = values
      @parent = parent
      @association_definition = association_definition
      @includes_association_definitions = includes_association_definitions
      @inverse_of = @association_definition&.inverse_of
      @foreign_key = @association_definition&.foreign_key
    end

    def where(filter = {})
      raise ActasticRecord::InvalidQueryError, "Invalid where: #{filter}" unless valid_conditions?(:filter, filter)
      merge_values(:conditions => { :filter => filter.to_a })
    end

    def not(**must_not)
      raise ActasticRecord::InvalidQueryError, "Invalid not: #{must_not}" unless valid_conditions?(:must_not, must_not)
      merge_values(:conditions => { :must_not => must_not.to_a })
    end

    def or(**should)
      raise ActasticRecord::InvalidQueryError, "Invalid or: #{should}" unless valid_conditions?(:should, should)
      merge_values(:conditions => { :should => should.to_a })
    end

    def order(*implicit_clauses, **explicit_clauses)
      clauses = implicit_clauses.map { |clause| [clause, DEFAULT_ORDER_DIRECTION] }.to_h.merge(explicit_clauses)
      raise ActasticRecord::InvalidQueryError, "Invalid order: #{clauses}" unless valid_order?(clauses)
      merge_values(:sort => clauses)
    end

    ##
    # Uses +collapse+ query in Elasticsearch: https://www.elastic.co/guide/en/elasticsearch/reference/current/collapse-search-results.html
    # It groups search results on +key_field+, and retrieves either one "top hit" per group (default), or multiple inner results per group,
    # if +size+ is more than 1. The "top" results within each group are determined by +sort_field+.
    # +collapse+ query has the following limitations:
    # * it cannot be used with scroll, rescore or search_after
    # * the +key_field+ must be a single valued keyword or numeric field with doc_values activated.
    def collapse(key_field, sort_field, sort_order: :asc, size: 1)
      raise ActasticRecord::InvalidQueryError, "Invalid collapse field: #{key_field}" unless valid_collapse_field?(key_field)
      sort_clause = { sort_field => sort_order }
      raise ActasticRecord::InvalidQueryError, "Invalid order: #{sort_clause}" unless valid_order?(sort_clause)
      collapse_clause = { :field => key_field }
      if size > 1
        collapse_clause[:inner_hits] = {
          :name => 'inner',
          :size => size,
          :sort => sort_clause
        }
      end
      merge_values(:collapse => collapse_clause, :sort => sort_clause)
    end

    def limit(size)
      raise ActasticRecord::InvalidQueryError, "Invalid limit: #{size}" unless size.is_a?(Integer)
      merge_values(:size => size)
    end

    def offset(from)
      raise ActasticRecord::InvalidQueryError, "Invalid offset: #{from}" unless from.is_a?(Integer)
      merge_values(:from => from)
    end

    def gt(**range_filter)
      range_filter(:gt, **range_filter)
    end

    def gte(**range_filter)
      range_filter(:gte, **range_filter)
    end

    def lt(**range_filter)
      range_filter(:lt, **range_filter)
    end

    def lte(**range_filter)
      range_filter(:lte, **range_filter)
    end

    def find(id = nil, &block)
      return super if block_given?

      record =
        if id && values.empty?
          result = retrieve(id)
          result && new_from_es_doc(result)
        else
          find_by(:_id => id)
        end

      record || raise(ActasticRecord::RecordNotFound, "No #{klass.name} with id #{id.inspect}")
    end

    def find_by(**filter)
      where(filter).limit(1).first
    end

    def find_by!(**args)
      find_by(**args) || raise(ActasticRecord::RecordNotFound, "No #{klass.name} found")
    end

    def find_or_initialize_by(**args)
      find_by(**args) || new(**args)
    end

    def find_or_create_by(**args)
      find_by(**args) || create(**args)
    end

    def find_or_create_by!(**args)
      find_by(**args) || create!(**args)
    end

    def exists?(**filter)
      filter.key?(:id) ? where(:_id => filter[:id]).any? : where(filter).any?
    end

    def count(*args)
      return super if block_given? || args.any? || embedded_filter_value.any?
      total = document_count - from_value
      limited? ? [total, size_value].min : total
    end

    def size
      loaded? ? @records.length : count
    end

    def pluck(*attributes)
      map do |record|
        values = attributes.map { |attribute| record[attribute] }
        (attributes.size > 1) ? values : values.first
      end
    end

    def new(attributes = {})
      raise InvalidForeignKey if belongs_to_many?
      klass.new(new_object_values.merge(attributes)).tap do |obj|
        yield obj if block_given?
        cache_parent_if_matches!(obj)
        add_to_records_if_parent_matches!(obj)
      end
    end

    def create!(attributes = {})
      raise InvalidForeignKey if belongs_to_many?
      klass.create!(new_object_values.merge(attributes)).tap do |obj|
        cache_parent_if_matches!(obj)
        add_to_records_if_parent_matches!(obj)
      end
    end

    def create(attributes = {})
      raise InvalidForeignKey if belongs_to_many?
      klass.create(new_object_values.merge(attributes)).tap do |obj|
        cache_parent_if_matches!(obj)
        add_to_records_if_parent_matches!(obj)
      end
    end

    def first_or_initialize(attributes = {}, &block)
      first || new(attributes, &block)
    end

    def destroy_all
      find_each(&:destroy!)
    end

    def update_all(attributes)
      each { |record| record.set(attributes) }
    end

    def delete_all
      each(&:delete)
    end

    def reload
      reset
      load
      self
    end

    def reset
      @records = nil
      self
    end

    def includes(*association_names)
      @includes_association_definitions += association_names.map { |association_name| klass.association_definitions[association_name] }.compact
      self
    end

    def none
      Actastic::NullRelation.new(klass, :values => values)
    end

    def first_with_limit(hits = nil)
      limit(hits || 1).first_without_limit(*hits)
    end
    alias_method :first_without_limit, :first
    alias_method :first, :first_with_limit

    def find_each(&block)
      return enum_for(:find_each) { document_count } unless block_given?

      result_count = 0
      hits = search.fetch(:hits)

      # if this was a collapse query, we have to retrieve inner hits from each top-level hit and combine them
      hits[:hits] = hits.fetch(:hits).flat_map { |h| h.dig(:inner_hits, :inner, :hits, :hits) } if inner_hits?

      while (results = hits.fetch(:hits)).any?
        result_count += results.size

        # FIXME: results are being returned that don't match the inheritance_column.
        # We should probably filter on that when searching.
        objects = results.map { |result| new_from_es_doc(result) }

        embedded_filter_value.each_with_object({}) do |(field, value), out|
          if value.nil?
            out[field] = nil
          else
            out[field] ||= HashWithIndifferentAccess.new
            out[field].merge!(value) unless value.nil?
          end
        end.each do |field, value|
          objects.select! do |object|
            if object[field].nil?
              value.nil?
            else
              Array.wrap(object[field]).any? { |embedded_object| embedded_object >= value }
            end
          end
        end

        @includes_association_definitions.each do |includes_association_definition|
          # TODO: implement other association types (has_one, belongs_to, etc)
          if includes_association_definition.type == :has_many && includes_association_definition.inverse_of && includes_association_definition.foreign_key
            # Fetch the associated records for all objects in this batch
            association_relation = includes_association_definition.klass
              .relation(:association_definition => includes_association_definition)
              .where(includes_association_definition.foreign_key => objects.map(&:id))
            objects.each do |object|
              # Get a reference to the association relation (which will stay in the object's cache)
              relation = object.public_send(includes_association_definition.name)
              # Put the relation into "loaded" state without running query (also makes add_to_records_if_parent_matches! work)
              relation.force_load!
              association_relation.each do |association_object|
                # Cache both directions (object <-> association_object) if foreign keys match
                relation.cache_parent_if_matches!(association_object)
                relation.add_to_records_if_parent_matches!(association_object)
              end
            end
          end
        end

        objects.each(&block)

        break if limited? || results.size < size_value || (result_count + from_value) >= es_version_support.total_hits(hits.fetch(:total)) || collapsed?

        hits = search(:search_after => results.last.fetch(:sort)).fetch(:hits)
      end
    end

    def <<(*objects)
      objects = Array(objects).flatten

      raise ActasticRecord::InvalidForeignKey unless belongs_to_many?
      raise ActasticRecord::InvalidForeignKey if objects.any? { |obj| !obj.is_a?(klass) }

      @parent.atomic_array_add!(@foreign_key, objects.map(&:id))
    end

    def delete(*objects)
      objects = Array(objects).flatten

      raise ActasticRecord::InvalidForeignKey unless belongs_to_many?
      raise ActasticRecord::InvalidForeignKey if objects.any? { |obj| !obj.is_a?(klass) }

      @parent.atomic_array_remove!(@foreign_key, objects.map(&:id))
    end

    def to_a
      load
      @records
    end

    def load
      @records ||= find_each.to_a
      self
    end

    def force_load!(*records)
      raise 'already loaded' if loaded?
      @records = records
    end

    def es_version_support
      Actastic.configuration.es_version_support
    end

    alias_method :to_ary, :to_a
    alias_method :in, :where
    alias_method :excludes, :not
    alias_method :nin, :not
    alias_method :ne, :not
    alias_method :build, :new

    delegate :+, :last, :empty?, :each, :length, :to => :to_a

    protected

    attr_reader :values

    def merge_values(**other_values)
      conditions = values.fetch(:conditions, {})
      other_conditions = other_values.fetch(:conditions, {})
      merged_conditions = conditions.merge(other_conditions) { |_type, a, b| a + b }
      self.class.new(
        klass,
        :values => values.deep_merge(**other_values).merge(:conditions => merged_conditions),
        :parent => @parent,
        :association_definition => @association_definition,
        :includes_association_definitions => @includes_association_definitions
      )
    end

    def loaded?
      !!@records
    end

    def foreign_key_matches?(obj)
      if @foreign_key && @parent && belongs_to_many?
        @parent.public_send(@foreign_key).include?(obj.id)
      elsif @foreign_key && @parent
        obj.public_send(@foreign_key) == @parent.id
      end
    end

    def cache_parent_if_matches!(obj)
      if @inverse_of && !belongs_to_many? && foreign_key_matches?(obj)
        obj.cache_association!(@inverse_of, @parent)
      end
    end

    def add_to_records_if_parent_matches!(obj)
      if loaded? && foreign_key_matches?(obj)
        @records << obj
      end
    end

    private

    attr_reader :klass

    def belongs_to_many?
      @association_definition&.type == :belongs_to_many
    end

    def valid_field?(field)
      field.is_a?(Symbol) || field.is_a?(String)
    end

    def valid_value?(value)
      Array.wrap(value).all? { |v| VALID_CONDITION_TYPES.any? { |t| v.is_a?(t) } }
    end

    def valid_conditions?(type, conditions)
      conditions.all? do |field, value|
        if klass.embedded_field?(field)
          valid = valid_field?(field)
          valid &&= value.is_a?(Hash) && valid_conditions?(type, value) unless value.nil?
          valid
        else
          valid_field?(field) && valid_value?(value)
        end
      end
    end

    def valid_collapse_field?(field)
      valid_field?(field) && VALID_COLLAPSE_FIELD_TYPES.include?(klass.schema.type_for(field))
    end

    def valid_range_field?(field)
      valid_field?(field) && VALID_RANGE_FIELD_TYPES.include?(klass.schema.type_for(field))
    end

    def valid_range_value?(value)
      VALID_RANGE_TYPES.any? { |range_type| value.is_a?(range_type) }
    end

    def valid_range_conditions?(conditions)
      conditions.all? { |field, value| valid_range_field?(field) && valid_range_value?(value) }
    end

    def valid_order?(clauses)
      clauses.all? do |field, direction|
        (field.is_a?(Symbol) || field.is_a?(String)) && VALID_ORDER_DIRECTIONS.include?(direction)
      end
    end

    def limited?
      @values.key?(:size)
    end

    def collapsed?
      @values.key?(:collapse)
    end

    def inner_hits?
      collapsed? && collapse_value.key?(:inner_hits)
    end

    def embedded_filter_value
      filter_value.select { |field, _value| klass.embedded_field?(field) }
    end

    def query_value
      conditions = @values.fetch(:conditions, {})
      if klass.schema.fields.include?(:type_) && klass.superclass != Object
        filter_conditions = conditions.fetch(:filter, [])
        filter_conditions += [[:type_, klass.descendants.map(&:name).push(klass.name)]]
        conditions = conditions.merge(:filter => filter_conditions)
      end
      {
        :bool => conditions.each_with_object({}) do |(type, clauses), bool|
          bool[:minimum_should_match] = 1 if type == :should
          if type == :range
            clauses.each do |range_type, filters|
              filters.each do |field, value|
                bool[:filter] ||= []
                if klass.schema.type_for(field) == 'date'
                  value = coerce_date(value)
                end
                bool[:filter] << { :range => { field => { range_type => value } } }
              end
            end
          else
            clauses.each do |field, value|
              # FIXME: array value including `nil`
              bool[type] ||= []
              if value.nil?
                bool[type] << { :bool => { :must_not => { :exists => { :field => field } } } }
              elsif value.is_a?(Range)
                begin_value = value.begin
                end_value = value.end
                if klass.schema.type_for(field) == 'date'
                  begin_value = coerce_date(begin_value)
                  end_value = coerce_date(end_value)
                end
                end_type = value.exclude_end? ? :lt : :lte
                bool[type] << { :range => { field => { :gte => begin_value, end_type => end_value } } }
              elsif klass.embedded_field?(field)
                value.each do |embedded_field, embedded_value|
                  bool[type] << { :terms => { "#{field}.#{embedded_field}" => Array.wrap(embedded_value) } }
                end
              elsif klass.encrypted_field?(field)
                bool[type] << { :terms => { "#{field}.hash" => Array.wrap(value).map { |v| Digest::SHA2.hexdigest(v) } } }
              else
                bool[type] << { :terms => { field => Array.wrap(value) } }
              end
            end
          end
        end
      }
    end

    def collapse_value
      @values.fetch(:collapse, {})
    end

    def filter_value
      @values.fetch(:conditions, {}).fetch(:filter, [])
    end

    def new_object_values
      # do not set the id for the new object
      filter_value.reject { |field, _value| field == :id }.to_h.tap do |new_object_values|
        if @foreign_key && @parent && !belongs_to_many?
          new_object_values[@foreign_key] = @parent.id
        end
      end
    end

    def sort_value
      @values.fetch(:sort, {}).map { |f, d| { f => d } }
    end

    def size_value
      @values.fetch(:size, DEFAULT_SIZE_VALUE)
    end

    def from_value
      @values.fetch(:from, 0)
    end

    def range_filter(range_type, **range_filter)
      raise ActasticRecord::InvalidQueryError, "Invalid #{range_type}: #{range_filter}" unless valid_range_conditions?(range_filter)
      merge_values(:conditions => { :range => [[range_type, range_filter]] })
    end

    # APM instrumentation for a given block of code
    def instrument(action, **context, &block)
      Actastic::ApmHelpers.instrument(klass, action, :context => context, &block)
    end

    def retrieve(id)
      instrument('retrieve', :id => id) do
        klass.schema.index.retrieve(id)
      end
    end

    def search(**options)
      search_opts = options.merge(
        :query => query_value,
        :sort => sort_value + %i(_doc),
        :size => size_value
      )

      search_opts = search_opts.merge(:from => from_value) if options[:search_after].blank?
      search_opts = search_opts.merge(:collapse => collapse_value) if collapsed?

      instrument('find', **search_opts) do
        klass.schema.index.search(
          search_opts.merge(
            :seq_no_primary_term => true,
            :request_cache => true
          )
        )
      end
    end

    def document_count
      instrument('count', :query => query_value) do
        klass.schema.index.document_count(query_value)
      end
    end

    def new_from_es_doc(es_doc)
      out = klass.new_from_es_doc(es_doc)
      cache_parent_if_matches!(out)
      out
    end

    # This is a naive implementation of how ActiveRecord includes class methods into relational scopes
    # The more comprehensive solution is explained in this StackOverflow post: https://stackoverflow.com/a/24767549
    def method_missing(m, *args, **kwargs, &block)
      return super unless klass.respond_to?(m)
      klass.current_scope.bind(self) do
        klass.public_send(m, *args, **kwargs, &block)
      end
    end

    def respond_to_missing?(method_name, include_private = false)
      klass.respond_to?(method_name, include_private) || super
    end

    Actastic::Stats.measure_methods(
      self,
      'actastic.relation',
      :search,
      :document_count
    )
  end
end
