"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDetectionRuleFromBenchmarkRule = void 0;
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/constants");
var _create_detection_rule = require("../../../common/api/create_detection_rule");
var _detection_rules = require("../../../../common/utils/detection_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_RULE_RISK_SCORE = 0;
const DEFAULT_RULE_SEVERITY = 'low';
const DEFAULT_RULE_ENABLED = true;
const DEFAULT_RULE_AUTHOR = 'Elastic';
const DEFAULT_RULE_LICENSE = 'Elastic License v2';
const DEFAULT_MAX_ALERTS_PER_RULE = 100;
const ALERT_SUPPRESSION_FIELD = 'resource.id';
const ALERT_TIMESTAMP_FIELD = 'event.ingested';
const DEFAULT_INVESTIGATION_FIELDS = {
  field_names: ['resource.name', 'resource.id', 'resource.type', 'resource.sub_type']
};
var AlertSuppressionMissingFieldsStrategy = /*#__PURE__*/function (AlertSuppressionMissingFieldsStrategy) {
  AlertSuppressionMissingFieldsStrategy["DoNotSuppress"] = "doNotSuppress";
  AlertSuppressionMissingFieldsStrategy["Suppress"] = "suppress";
  return AlertSuppressionMissingFieldsStrategy;
}(AlertSuppressionMissingFieldsStrategy || {});
const convertReferencesLinksToArray = input => {
  if (!input) {
    return [];
  }
  // Match all URLs in the input string using a regular expression
  const matches = input.match(/(https?:\/\/\S+)/g);
  if (!matches) {
    return [];
  }

  // Remove the numbers and new lines
  return matches.map(link => link.replace(/^\d+\. /, '').replace(/\n/g, ''));
};
const generateFindingsRuleQuery = benchmarkRule => {
  const currentTimestamp = new Date().toISOString();
  return `rule.benchmark.rule_number: "${benchmarkRule.benchmark.rule_number}"
    AND rule.benchmark.id: "${benchmarkRule.benchmark.id}"
    AND result.evaluation: "failed"
    AND event.ingested >= "${currentTimestamp}"`;
};

/*
 * Creates a detection rule from a Benchmark rule
 */
const createDetectionRuleFromBenchmarkRule = async (http, benchmarkRule) => {
  var _benchmarkRule$benchm;
  if (!((_benchmarkRule$benchm = benchmarkRule.benchmark) !== null && _benchmarkRule$benchm !== void 0 && _benchmarkRule$benchm.posture_type)) {
    throw new Error(_i18n.i18n.translate('xpack.csp.createDetectionRuleFromBenchmarkRule.createRuleErrorMessage', {
      defaultMessage: 'Rule creation is currently only available for Elastic findings'
    }));
  }
  return await (0, _create_detection_rule.createDetectionRule)({
    http,
    rule: {
      type: 'query',
      language: 'kuery',
      license: DEFAULT_RULE_LICENSE,
      author: [DEFAULT_RULE_AUTHOR],
      filters: [],
      false_positives: [],
      risk_score: DEFAULT_RULE_RISK_SCORE,
      risk_score_mapping: [],
      severity: DEFAULT_RULE_SEVERITY,
      severity_mapping: [],
      threat: [],
      interval: '1h',
      from: `now-${_cloudSecurityPostureCommon.LATEST_FINDINGS_RETENTION_POLICY}`,
      to: 'now',
      max_signals: DEFAULT_MAX_ALERTS_PER_RULE,
      timestamp_override: ALERT_TIMESTAMP_FIELD,
      timestamp_override_fallback_disabled: false,
      actions: [],
      enabled: DEFAULT_RULE_ENABLED,
      alert_suppression: {
        group_by: [ALERT_SUPPRESSION_FIELD],
        missing_fields_strategy: AlertSuppressionMissingFieldsStrategy.Suppress
      },
      index: [_constants.FINDINGS_INDEX_PATTERN],
      query: generateFindingsRuleQuery(benchmarkRule),
      references: convertReferencesLinksToArray(benchmarkRule.references),
      name: benchmarkRule.name,
      description: benchmarkRule.rationale,
      tags: (0, _detection_rules.generateBenchmarkRuleTags)(benchmarkRule),
      investigation_fields: DEFAULT_INVESTIGATION_FIELDS,
      note: benchmarkRule.remediation
    }
  });
};
exports.createDetectionRuleFromBenchmarkRule = createDetectionRuleFromBenchmarkRule;