"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CoreScopedHistory = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Core's internal implementation of {@link ScopedHistory}
 *
 * @internal Only exposed publicly for testing purpose.
 */
class CoreScopedHistory {
  constructor(parentHistory, _basePath) {
    /**
     * Tracks whether or not the user has left this history's scope. All methods throw errors if called after scope has
     * been left.
     */
    (0, _defineProperty2.default)(this, "isActive", true);
    /**
     * All active listeners on this history instance.
     */
    (0, _defineProperty2.default)(this, "listeners", new Set());
    /**
     * Array of the local history stack. Only stores {@link Location.key} to use tracking an index of the current
     * position of the window in the history stack.
     */
    (0, _defineProperty2.default)(this, "locationKeys", []);
    /**
     * The key of the current position of the window in the history stack.
     */
    (0, _defineProperty2.default)(this, "currentLocationKeyIndex", 0);
    /**
     * Array of the current {@link block} unregister callbacks
     */
    (0, _defineProperty2.default)(this, "blockUnregisterCallbacks", new Set());
    /**
     * Creates a `ScopedHistory` for a subpath of this `ScopedHistory`. Useful for applications that may have sub-apps
     * that do not need access to the containing application's history.
     *
     * @param basePath the URL path scope for the sub history
     */
    (0, _defineProperty2.default)(this, "createSubHistory", basePath => {
      return new CoreScopedHistory(this, basePath);
    });
    /**
     * Pushes a new location onto the history stack. If there are forward entries in the stack, they will be removed.
     *
     * @param pathOrLocation a string or location descriptor
     * @param state
     */
    (0, _defineProperty2.default)(this, "push", (pathOrLocation, state) => {
      this.verifyActive();
      if (typeof pathOrLocation === 'string') {
        this.parentHistory.push(this.prependBasePath(pathOrLocation), state);
      } else {
        this.parentHistory.push(this.prependBasePath(pathOrLocation));
      }
    });
    /**
     * Replaces the current location in the history stack. Does not remove forward or backward entries.
     *
     * @param pathOrLocation a string or location descriptor
     * @param state
     */
    (0, _defineProperty2.default)(this, "replace", (pathOrLocation, state) => {
      this.verifyActive();
      if (typeof pathOrLocation === 'string') {
        this.parentHistory.replace(this.prependBasePath(pathOrLocation), state);
      } else {
        this.parentHistory.replace(this.prependBasePath(pathOrLocation));
      }
    });
    /**
     * Send the user forward or backwards in the history stack.
     *
     * @param n number of positions in the stack to go. Negative numbers indicate number of entries backward, positive
     *          numbers for forwards. If passed 0, the current location will be reloaded. If `n` exceeds the number of
     *          entries available, this is a no-op.
     */
    (0, _defineProperty2.default)(this, "go", n => {
      this.verifyActive();
      if (n === 0) {
        this.parentHistory.go(n);
      } else if (n < 0) {
        if (this.currentLocationKeyIndex + 1 + n >= 1) {
          this.parentHistory.go(n);
        }
      } else if (n <= this.currentLocationKeyIndex + this.locationKeys.length - 1) {
        this.parentHistory.go(n);
      }
      // no-op if no conditions above are met
    });
    /**
     * Send the user one location back in the history stack. Equivalent to calling
     * {@link ScopedHistory.go | ScopedHistory.go(-1)}. If no more entries are available backwards, this is a no-op.
     */
    (0, _defineProperty2.default)(this, "goBack", () => {
      this.verifyActive();
      this.go(-1);
    });
    /**
     * Send the user one location forward in the history stack. Equivalent to calling
     * {@link ScopedHistory.go | ScopedHistory.go(1)}. If no more entries are available forwards, this is a no-op.
     */
    (0, _defineProperty2.default)(this, "goForward", () => {
      this.verifyActive();
      this.go(1);
    });
    /**
     * Add a block prompt requesting user confirmation when navigating away from the current page.
     */
    (0, _defineProperty2.default)(this, "block", prompt => {
      this.verifyActive();
      const unregisterCallback = this.parentHistory.block(prompt);
      this.blockUnregisterCallbacks.add(unregisterCallback);
      return () => {
        this.blockUnregisterCallbacks.delete(unregisterCallback);
        unregisterCallback();
      };
    });
    /**
     * Adds a listener for location updates.
     *
     * @param listener a function that receives location updates.
     * @returns an function to unsubscribe the listener.
     */
    (0, _defineProperty2.default)(this, "listen", listener => {
      this.verifyActive();
      this.listeners.add(listener);
      return () => {
        this.listeners.delete(listener);
      };
    });
    /**
     * Creates an href (string) to the location.
     * If `prependBasePath` is true (default), it will prepend the location's path with the scoped history basePath.
     *
     * @param location
     * @param prependBasePath
     */
    (0, _defineProperty2.default)(this, "createHref", (location, {
      prependBasePath = true
    } = {}) => {
      this.verifyActive();
      if (prependBasePath) {
        location = this.prependBasePath(location);
        if (location.pathname === undefined) {
          // we always want to create an url relative to the basePath
          // so if pathname is not present, we use the history's basePath as default
          // we are doing that here because `prependBasePath` should not
          // alter pathname for other method calls
          location.pathname = this.basePath;
        }
      }
      return this.parentHistory.createHref(location);
    });
    this.parentHistory = parentHistory;
    this.basePath = _basePath;
    const parentPath = this.parentHistory.location.pathname;
    if (!parentPath.startsWith(_basePath)) {
      throw new Error(`Browser location [${parentPath}] is not currently in expected basePath [${_basePath}]`);
    }
    this.locationKeys.push(this.parentHistory.location.key);
    this.setupHistoryListener();
  }
  /**
   * The number of entries in the history stack, including all entries forwards and backwards from the current location.
   */
  get length() {
    return this.locationKeys.length;
  }

  /**
   * The current location of the history stack.
   */
  get location() {
    return this.stripBasePath(this.parentHistory.location);
  }

  /**
   * The last action dispatched on the history stack.
   */
  get action() {
    return this.parentHistory.action;
  }
  /**
   * Prepends the scoped base path to the Path or Location
   */
  prependBasePath(pathOrLocation) {
    if (typeof pathOrLocation === 'string') {
      return this.prependBasePathToString(pathOrLocation);
    } else {
      return {
        ...pathOrLocation,
        pathname: pathOrLocation.pathname !== undefined ? this.prependBasePathToString(pathOrLocation.pathname) : undefined
      };
    }
  }

  /**
   * Prepends the base path to string.
   */
  prependBasePathToString(path) {
    return path.length ? `${this.basePath}/${path}`.replace(/\/{2,}/g, '/') : this.basePath;
  }

  /**
   * Removes the base path from a location.
   */
  stripBasePath(location) {
    return {
      ...location,
      pathname: location.pathname.replace(new RegExp(`^${this.basePath}`), '')
    };
  }

  /** Called on each public method to ensure that we have not fallen out of scope yet. */
  verifyActive() {
    if (!this.isActive) {
      throw new Error(`ScopedHistory instance has fell out of navigation scope for basePath: ${this.basePath}`);
    }
  }

  /**
   * Sets up the listener on the parent history instance used to follow navigation updates and track our internal
   * state. Also forwards events to child listeners with the base path stripped from the location.
   */
  setupHistoryListener() {
    const unlisten = this.parentHistory.listen((location, action) => {
      // If the user navigates outside the scope of this basePath, tear it down.
      if (!location.pathname.startsWith(this.basePath)) {
        unlisten();
        this.isActive = false;
        for (const unregisterBlock of this.blockUnregisterCallbacks) {
          unregisterBlock();
        }
        this.blockUnregisterCallbacks.clear();
        return;
      }

      /**
       * Track location keys using the same algorithm the browser uses internally.
       * - On PUSH, remove all items that came after the current location and append the new location.
       * - On POP, set the current location, but do not change the entries.
       * - On REPLACE, override the location for the current index with the new location.
       */
      if (action === 'PUSH') {
        this.locationKeys = [...this.locationKeys.slice(0, this.currentLocationKeyIndex + 1), location.key];
        this.currentLocationKeyIndex = this.locationKeys.indexOf(location.key); // should always be the last index
      } else if (action === 'POP') {
        this.currentLocationKeyIndex = this.locationKeys.indexOf(location.key);
      } else if (action === 'REPLACE') {
        this.locationKeys[this.currentLocationKeyIndex] = location.key;
      } else {
        throw new Error(`Unrecognized history action: ${action}`);
      }
      [...this.listeners].forEach(listener => {
        listener(this.stripBasePath(location), action);
      });
    });
  }
}
exports.CoreScopedHistory = CoreScopedHistory;