"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.replaceCustomField = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _api = require("../../../common/types/api");
var _authorization = require("../../authorization");
var _error = require("../../common/error");
var _runtime_types = require("../../common/runtime_types");
var _domain = require("../../../common/types/domain");
var _validators = require("./validators");
var _validators2 = require("../../common/validators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Updates the specified cases with new values
 *
 * @ignore
 */
const replaceCustomField = async ({
  caseId,
  customFieldId,
  request
}, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      userActionService
    },
    user,
    logger,
    authorization
  } = clientArgs;
  try {
    var _configurations$, _updatedCase$attribut;
    const {
      value,
      caseVersion
    } = request;
    (0, _runtime_types.decodeWithExcessOrThrow)(_api.CustomFieldPutRequestRt)(request);
    const caseToUpdate = await caseService.getCase({
      id: caseId
    });
    if (caseToUpdate.version !== caseVersion) {
      throw _boom.default.conflict(`This case ${caseToUpdate.id} has been updated. Please refresh before saving additional updates.`);
    }
    const configurations = await casesClient.configure.get({
      owner: caseToUpdate.attributes.owner
    });
    await authorization.ensureAuthorized({
      entities: [{
        owner: caseToUpdate.attributes.owner,
        id: caseToUpdate.id
      }],
      operation: _authorization.Operations.updateCase
    });
    const foundCustomField = (_configurations$ = configurations[0]) === null || _configurations$ === void 0 ? void 0 : _configurations$.customFields.find(item => item.key === customFieldId);
    if (!foundCustomField) {
      throw _boom.default.badRequest('cannot find custom field');
    }
    (0, _validators.validateCustomFieldTypesInRequest)({
      requestCustomFields: [{
        value,
        type: foundCustomField.type,
        key: customFieldId
      }],
      customFieldsConfiguration: configurations[0].customFields
    });
    if (value == null && foundCustomField.required) {
      throw _boom.default.badRequest('Custom field value cannot be null or undefined.');
    }
    const customFieldsToUpdate = [{
      value,
      type: foundCustomField.type,
      key: customFieldId
    }, ...caseToUpdate.attributes.customFields.filter(field => field.key !== customFieldId)];
    const decodedCustomFields = (0, _runtime_types.decodeWithExcessOrThrow)(_api.CaseRequestCustomFieldsRt)(customFieldsToUpdate);
    const updatedAt = new Date().toISOString();
    const patchCasesPayload = {
      caseId,
      originalCase: caseToUpdate,
      updatedAttributes: {
        customFields: decodedCustomFields,
        updated_at: updatedAt,
        updated_by: user
      },
      version: caseVersion
    };
    const userActionsDict = userActionService.creator.buildUserActions({
      updatedCases: {
        cases: [patchCasesPayload]
      },
      user
    });
    await (0, _validators2.validateMaxUserActions)({
      caseId,
      userActionService,
      userActionsToAdd: 1
    });
    const updatedCase = await caseService.patchCase({
      ...patchCasesPayload,
      refresh: false
    });
    const updatedCustomField = (_updatedCase$attribut = updatedCase.attributes.customFields) === null || _updatedCase$attribut === void 0 ? void 0 : _updatedCase$attribut.find(cf => cf.key === customFieldId);
    if (!updatedCustomField) {
      throw new Error('Cannot find updated custom field.');
    }
    const builtUserActions = userActionsDict != null ? Object.keys(userActionsDict).reduce((acc, key) => {
      return [...acc, ...userActionsDict[key]];
    }, []) : [];
    await userActionService.creator.bulkCreateUpdateCase({
      builtUserActions
    });
    return (0, _runtime_types.decodeOrThrow)(_domain.CaseCustomFieldRt)(updatedCustomField);
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to replace customField, id: ${customFieldId} of case: ${caseId} version:${request.caseVersion} : ${error}`,
      error,
      logger
    });
  }
};
exports.replaceCustomField = replaceCustomField;