"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUsageMetricsHandler = void 0;
exports.transformMetricsData = transformMetricsData;
var _fp = require("lodash/fp");
var _utils = require("../../../common/utils");
var _error_handler = require("../error_handler");
var _utils2 = require("../../utils");
var _services = require("../../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatStringParams = value => typeof value === 'string' ? [value] : value;
const getUsageMetricsHandler = dataUsageContext => {
  const logger = dataUsageContext.logFactory.get('usageMetricsRoute');
  return async (context, request, response) => {
    try {
      var _momentDateParser, _momentDateParser2;
      const core = await context.core;
      const esClient = core.elasticsearch.client.asCurrentUser;
      const getDataStreams = name => esClient.indices.getDataStream({
        name,
        expand_wildcards: 'all'
      });
      logger.debug(`Retrieving usage metrics`);
      const {
        from,
        to,
        metricTypes,
        dataStreams: requestDsNames
      } = request.body;

      // parse date strings to validate
      const parsedFrom = (_momentDateParser = (0, _utils.momentDateParser)(from)) === null || _momentDateParser === void 0 ? void 0 : _momentDateParser.toISOString();
      const parsedTo = (_momentDateParser2 = (0, _utils.momentDateParser)(to)) === null || _momentDateParser2 === void 0 ? void 0 : _momentDateParser2.toISOString();
      if (!parsedFrom || !parsedTo) {
        const customErrorMessage = `[request body.${!parsedTo ? 'to' : 'from'}] Invalid date range ${!parsedTo ? to : from} is out of range`;
        return (0, _error_handler.errorHandler)(logger, response, new _utils2.CustomHttpRequestError(customErrorMessage, 400));
      }

      // redundant check as we don't allow making requests via UI without data streams,
      // but it's here to make sure the request body is validated before requesting metrics from auto-ops
      if (!(requestDsNames !== null && requestDsNames !== void 0 && requestDsNames.length)) {
        return (0, _error_handler.errorHandler)(logger, response, new _utils2.CustomHttpRequestError('[request body.dataStreams]: no data streams selected', 400));
      }
      let dataStreamsResponse;
      try {
        if (requestDsNames.length <= 50) {
          logger.debug(`Retrieving usage metrics`);
          const {
            data_streams: dataStreams
          } = await getDataStreams(requestDsNames);
          dataStreamsResponse = dataStreams;
        } else {
          logger.debug(`Retrieving usage metrics in chunks of 50`);
          // Attempt to fetch data streams in chunks of 50
          const dataStreamsChunks = Math.ceil(requestDsNames.length / 50);
          const chunkedDsLists = (0, _fp.chunk)(dataStreamsChunks, requestDsNames);
          const chunkedDataStreams = await Promise.all(chunkedDsLists.map(dsList => getDataStreams(dsList)));
          dataStreamsResponse = chunkedDataStreams.flatMap(ds => ds.data_streams);
        }
      } catch (error) {
        return (0, _error_handler.errorHandler)(logger, response, new _utils2.CustomHttpRequestError('Failed to retrieve data streams', 400));
      }
      const dataUsageService = new _services.DataUsageService(logger);
      const metrics = await dataUsageService.getMetrics({
        from: parsedFrom,
        to: parsedTo,
        metricTypes: formatStringParams(metricTypes),
        dataStreams: formatStringParams(dataStreamsResponse.map(ds => ds.name))
      });
      const body = transformMetricsData(metrics);
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`Error retrieving usage metrics: ${error.message}`);
      return (0, _error_handler.errorHandler)(logger, response, error);
    }
  };
};
exports.getUsageMetricsHandler = getUsageMetricsHandler;
function transformMetricsData(data) {
  return Object.fromEntries(Object.entries(data).map(([metricType, series]) => [metricType, series.map(metricSeries => ({
    name: metricSeries.name,
    data: metricSeries.data.map(([timestamp, value]) => ({
      x: timestamp,
      y: value
    }))
  }))]));
}