/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.csv;

import java.io.Closeable;
import java.io.IOException;
import org.apache.commons.csv.CSVException;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.ExtendedBufferedReader;
import org.apache.commons.csv.Token;

final class Lexer
implements Closeable {
    private static final String CR_STRING = Character.toString('\r');
    private static final String LF_STRING = Character.toString('\n');
    private final char[] delimiter;
    private final char[] delimiterBuf;
    private final char[] escapeDelimiterBuf;
    private final int escape;
    private final int quoteChar;
    private final int commentStart;
    private final boolean ignoreSurroundingSpaces;
    private final boolean ignoreEmptyLines;
    private final boolean lenientEof;
    private final boolean trailingData;
    private final ExtendedBufferedReader reader;
    private String firstEol;
    private boolean isLastTokenDelimiter;

    Lexer(CSVFormat format, ExtendedBufferedReader reader) {
        this.reader = reader;
        this.delimiter = format.getDelimiterCharArray();
        this.escape = this.nullToDisabled(format.getEscapeCharacter());
        this.quoteChar = this.nullToDisabled(format.getQuoteCharacter());
        this.commentStart = this.nullToDisabled(format.getCommentMarker());
        this.ignoreSurroundingSpaces = format.getIgnoreSurroundingSpaces();
        this.ignoreEmptyLines = format.getIgnoreEmptyLines();
        this.lenientEof = format.getLenientEof();
        this.trailingData = format.getTrailingData();
        this.delimiterBuf = new char[this.delimiter.length - 1];
        this.escapeDelimiterBuf = new char[2 * this.delimiter.length - 1];
    }

    private void appendNextEscapedCharacterToToken(Token token) throws IOException {
        if (this.isEscapeDelimiter()) {
            token.content.append(this.delimiter);
        } else {
            int unescaped = this.readEscape();
            if (unescaped == -1) {
                token.content.append((char)this.escape).append((char)this.reader.getLastChar());
            } else {
                token.content.append((char)unescaped);
            }
        }
    }

    @Override
    public void close() throws IOException {
        this.reader.close();
    }

    long getBytesRead() {
        return this.reader.getBytesRead();
    }

    long getCharacterPosition() {
        return this.reader.getPosition();
    }

    long getCurrentLineNumber() {
        return this.reader.getLineNumber();
    }

    String getFirstEol() {
        return this.firstEol;
    }

    boolean isClosed() {
        return this.reader.isClosed();
    }

    boolean isCommentStart(int ch) {
        return ch == this.commentStart;
    }

    boolean isDelimiter(int ch) throws IOException {
        this.isLastTokenDelimiter = false;
        if (ch != this.delimiter[0]) {
            return false;
        }
        if (this.delimiter.length == 1) {
            this.isLastTokenDelimiter = true;
            return true;
        }
        this.reader.peek(this.delimiterBuf);
        for (int i = 0; i < this.delimiterBuf.length; ++i) {
            if (this.delimiterBuf[i] == this.delimiter[i + 1]) continue;
            return false;
        }
        int count = this.reader.read(this.delimiterBuf, 0, this.delimiterBuf.length);
        this.isLastTokenDelimiter = count != -1;
        return this.isLastTokenDelimiter;
    }

    boolean isEndOfFile(int ch) {
        return ch == -1;
    }

    boolean isEscape(int ch) {
        return ch == this.escape;
    }

    boolean isEscapeDelimiter() throws IOException {
        this.reader.peek(this.escapeDelimiterBuf);
        if (this.escapeDelimiterBuf[0] != this.delimiter[0]) {
            return false;
        }
        for (int i = 1; i < this.delimiter.length; ++i) {
            if (this.escapeDelimiterBuf[2 * i] == this.delimiter[i] && this.escapeDelimiterBuf[2 * i - 1] == this.escape) continue;
            return false;
        }
        int count = this.reader.read(this.escapeDelimiterBuf, 0, this.escapeDelimiterBuf.length);
        return count != -1;
    }

    private boolean isMetaChar(int ch) {
        return ch == this.escape || ch == this.quoteChar || ch == this.commentStart;
    }

    boolean isQuoteChar(int ch) {
        return ch == this.quoteChar;
    }

    boolean isStartOfLine(int ch) {
        return ch == 10 || ch == 13 || ch == -2;
    }

    Token nextToken(Token token) throws IOException {
        int lastChar = this.reader.getLastChar();
        int c = this.reader.read();
        boolean eol = this.readEndOfLine(c);
        if (this.ignoreEmptyLines) {
            while (eol && this.isStartOfLine(lastChar)) {
                lastChar = c;
                c = this.reader.read();
                eol = this.readEndOfLine(c);
                if (!this.isEndOfFile(c)) continue;
                token.type = Token.Type.EOF;
                return token;
            }
        }
        if (this.isEndOfFile(lastChar) || !this.isLastTokenDelimiter && this.isEndOfFile(c)) {
            token.type = Token.Type.EOF;
            return token;
        }
        if (this.isStartOfLine(lastChar) && this.isCommentStart(c)) {
            String line = this.reader.readLine();
            if (line == null) {
                token.type = Token.Type.EOF;
                return token;
            }
            String comment = line.trim();
            token.content.append(comment);
            token.type = Token.Type.COMMENT;
            return token;
        }
        while (token.type == Token.Type.INVALID) {
            if (this.ignoreSurroundingSpaces) {
                while (Character.isWhitespace((char)c) && !this.isDelimiter(c) && !eol) {
                    c = this.reader.read();
                    eol = this.readEndOfLine(c);
                }
            }
            if (this.isDelimiter(c)) {
                token.type = Token.Type.TOKEN;
                continue;
            }
            if (eol) {
                token.type = Token.Type.EORECORD;
                continue;
            }
            if (this.isQuoteChar(c)) {
                this.parseEncapsulatedToken(token);
                continue;
            }
            if (this.isEndOfFile(c)) {
                token.type = Token.Type.EOF;
                token.isReady = true;
                continue;
            }
            this.parseSimpleToken(token, c);
        }
        return token;
    }

    private int nullToDisabled(Character c) {
        return c == null ? -2 : (int)c.charValue();
    }

    private Token parseEncapsulatedToken(Token token) throws IOException {
        token.isQuoted = true;
        long startLineNumber = this.getCurrentLineNumber();
        while (true) {
            int c;
            if (this.isQuoteChar(c = this.reader.read())) {
                if (this.isQuoteChar(this.reader.peek())) {
                    c = this.reader.read();
                    token.content.append((char)c);
                    continue;
                }
                while (true) {
                    if (this.isDelimiter(c = this.reader.read())) {
                        token.type = Token.Type.TOKEN;
                        return token;
                    }
                    if (this.isEndOfFile(c)) {
                        token.type = Token.Type.EOF;
                        token.isReady = true;
                        return token;
                    }
                    if (this.readEndOfLine(c)) {
                        token.type = Token.Type.EORECORD;
                        return token;
                    }
                    if (this.trailingData) {
                        token.content.append((char)c);
                        continue;
                    }
                    if (!Character.isWhitespace((char)c)) break;
                }
                throw new CSVException("Invalid character between encapsulated token and delimiter at line: %,d, position: %,d", this.getCurrentLineNumber(), this.getCharacterPosition());
            }
            if (this.isEscape(c)) {
                this.appendNextEscapedCharacterToToken(token);
                continue;
            }
            if (this.isEndOfFile(c)) {
                if (this.lenientEof) {
                    token.type = Token.Type.EOF;
                    token.isReady = true;
                    return token;
                }
                throw new CSVException("(startline %,d) EOF reached before encapsulated token finished", startLineNumber);
            }
            token.content.append((char)c);
        }
    }

    private Token parseSimpleToken(Token token, int ch) throws IOException {
        int cur = ch;
        while (true) {
            if (this.readEndOfLine(cur)) {
                token.type = Token.Type.EORECORD;
                break;
            }
            if (this.isEndOfFile(cur)) {
                token.type = Token.Type.EOF;
                token.isReady = true;
                break;
            }
            if (this.isDelimiter(cur)) {
                token.type = Token.Type.TOKEN;
                break;
            }
            if (this.isEscape(cur)) {
                this.appendNextEscapedCharacterToToken(token);
            } else {
                token.content.append((char)cur);
            }
            cur = this.reader.read();
        }
        if (this.ignoreSurroundingSpaces) {
            this.trimTrailingSpaces(token.content);
        }
        return token;
    }

    boolean readEndOfLine(int ch) throws IOException {
        int cur = ch;
        if (cur == 13 && this.reader.peek() == 10) {
            cur = this.reader.read();
            if (this.firstEol == null) {
                this.firstEol = "\r\n";
            }
        }
        if (this.firstEol == null) {
            if (cur == 10) {
                this.firstEol = LF_STRING;
            } else if (cur == 13) {
                this.firstEol = CR_STRING;
            }
        }
        return cur == 10 || cur == 13;
    }

    int readEscape() throws IOException {
        int ch = this.reader.read();
        switch (ch) {
            case 114: {
                return 13;
            }
            case 110: {
                return 10;
            }
            case 116: {
                return 9;
            }
            case 98: {
                return 8;
            }
            case 102: {
                return 12;
            }
            case 8: 
            case 9: 
            case 10: 
            case 12: 
            case 13: {
                return ch;
            }
            case -1: {
                throw new CSVException("EOF while processing escape sequence", new Object[0]);
            }
        }
        if (this.isMetaChar(ch)) {
            return ch;
        }
        return -1;
    }

    void trimTrailingSpaces(StringBuilder buffer) {
        int length;
        for (length = buffer.length(); length > 0 && Character.isWhitespace(buffer.charAt(length - 1)); --length) {
        }
        if (length != buffer.length()) {
            buffer.setLength(length);
        }
    }
}

