/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.node;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.TransportVersions;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.node.VersionInformation;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.util.StringLiteralDeduplicator;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.env.BuildVersion;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.NodeRoleSettings;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentFragment;
import org.elasticsearch.xcontent.XContentBuilder;

public class DiscoveryNode
implements Writeable,
ToXContentFragment {
    public static final String STATELESS_ENABLED_SETTING_NAME = "stateless.enabled";
    static final String COORDINATING_ONLY = "coordinating_only";
    public static final TransportVersion EXTERNAL_ID_VERSION = TransportVersions.V_8_3_0;
    public static final Comparator<DiscoveryNode> DISCOVERY_NODE_COMPARATOR = Comparator.comparing(DiscoveryNode::getName).thenComparing(DiscoveryNode::getId);
    private static final StringLiteralDeduplicator nodeStringDeduplicator = new StringLiteralDeduplicator();
    private final String nodeName;
    private final String nodeId;
    private final String ephemeralId;
    private final String hostName;
    private final String hostAddress;
    private final TransportAddress address;
    private final Map<String, String> attributes;
    private final VersionInformation versionInfo;
    private final SortedSet<DiscoveryNodeRole> roles;
    private final Set<String> roleNames;
    private final String externalId;
    private static final Writeable.Reader<String> readStringLiteral = s -> nodeStringDeduplicator.deduplicate(s.readString());

    public static boolean isStateless(Settings settings) {
        return settings.getAsBoolean(STATELESS_ENABLED_SETTING_NAME, false);
    }

    public static boolean hasRole(Settings settings, DiscoveryNodeRole role) {
        if (settings.hasValue("node.roles")) {
            return settings.getAsList("node.roles").contains(role.roleName());
        }
        return role.isEnabledByDefault(settings);
    }

    public static boolean isMasterNode(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.MASTER_ROLE);
    }

    public static boolean hasDataRole(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.DATA_ROLE);
    }

    public static boolean canContainData(Settings settings) {
        return DiscoveryNode.getRolesFromSettings(settings).stream().anyMatch(DiscoveryNodeRole::canContainData);
    }

    public static boolean isIngestNode(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.INGEST_ROLE);
    }

    public static boolean isRemoteClusterClient(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.REMOTE_CLUSTER_CLIENT_ROLE);
    }

    private static boolean isDedicatedFrozenRoles(Set<DiscoveryNodeRole> roles) {
        return roles.contains(DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE) && !roles.stream().filter(DiscoveryNodeRole::canContainData).anyMatch(r -> r != DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE);
    }

    public static boolean isDedicatedFrozenNode(Settings settings) {
        return DiscoveryNode.isDedicatedFrozenRoles(DiscoveryNode.getRolesFromSettings(settings));
    }

    public DiscoveryNode(@Nullable String nodeName, String nodeId, TransportAddress address, Map<String, String> attributes, Set<DiscoveryNodeRole> roles, @Nullable VersionInformation versionInfo) {
        this(nodeName, nodeId, UUIDs.randomBase64UUID(), address.address().getHostString(), address.getAddress(), address, attributes, roles, versionInfo);
    }

    public DiscoveryNode(@Nullable String nodeName, String nodeId, String ephemeralId, String hostName, String hostAddress, TransportAddress address, Map<String, String> attributes, Set<DiscoveryNodeRole> roles, @Nullable VersionInformation versionInfo) {
        this(nodeName, nodeId, ephemeralId, hostName, hostAddress, address, attributes, roles, versionInfo, null);
    }

    public DiscoveryNode(@Nullable String nodeName, String nodeId, String ephemeralId, String hostName, String hostAddress, TransportAddress address, Map<String, String> attributes, Set<DiscoveryNodeRole> roles, @Nullable VersionInformation versionInfo, @Nullable String externalId) {
        this.nodeName = nodeName != null ? nodeStringDeduplicator.deduplicate(nodeName) : "";
        this.nodeId = nodeStringDeduplicator.deduplicate(nodeId);
        this.ephemeralId = nodeStringDeduplicator.deduplicate(ephemeralId);
        this.hostName = nodeStringDeduplicator.deduplicate(hostName);
        assert (Strings.hasText(hostAddress));
        this.hostAddress = nodeStringDeduplicator.deduplicate(hostAddress);
        this.address = address;
        this.versionInfo = Objects.requireNonNullElse(versionInfo, VersionInformation.CURRENT);
        this.attributes = Map.copyOf(attributes);
        assert (DiscoveryNodeRole.roleNames().stream().noneMatch(attributes::containsKey)) : "Node roles must not be provided as attributes but saw attributes " + String.valueOf(attributes);
        TreeSet<DiscoveryNodeRole> sortedRoles = new TreeSet<DiscoveryNodeRole>();
        String[] roleNames = new String[roles.size()];
        int i = 0;
        for (DiscoveryNodeRole role : roles) {
            sortedRoles.add(role);
            roleNames[i++] = role.roleName();
        }
        this.roles = Collections.unmodifiableSortedSet(sortedRoles);
        this.roleNames = Set.of(roleNames);
        this.externalId = Objects.requireNonNullElse(externalId, this.nodeName);
    }

    public static DiscoveryNode createLocal(Settings settings, TransportAddress publishAddress, String nodeId) {
        Map<String, String> attributes = Node.NODE_ATTRIBUTES.getAsMap(settings);
        Set<DiscoveryNodeRole> roles = DiscoveryNode.getRolesFromSettings(settings);
        return new DiscoveryNode(Node.NODE_NAME_SETTING.get(settings), nodeId, UUIDs.randomBase64UUID(), publishAddress.address().getHostString(), publishAddress.getAddress(), publishAddress, attributes, roles, null, Node.NODE_EXTERNAL_ID_SETTING.get(settings));
    }

    public static Set<DiscoveryNodeRole> getRolesFromSettings(Settings settings) {
        return Set.copyOf((Collection)NodeRoleSettings.NODE_ROLES_SETTING.get(settings));
    }

    private static VersionInformation inferVersionInformation(Version version) {
        if (version.before(Version.V_8_10_0)) {
            return new VersionInformation(version, IndexVersion.getMinimumCompatibleIndexVersion(version.id), IndexVersion.fromId(version.id));
        }
        return new VersionInformation(version, IndexVersions.MINIMUM_COMPATIBLE, IndexVersion.current());
    }

    public DiscoveryNode(StreamInput in) throws IOException {
        this.nodeName = readStringLiteral.read(in);
        this.nodeId = readStringLiteral.read(in);
        this.ephemeralId = readStringLiteral.read(in);
        this.hostName = readStringLiteral.read(in);
        this.hostAddress = readStringLiteral.read(in);
        this.address = new TransportAddress(in);
        this.attributes = in.readImmutableMap(readStringLiteral, readStringLiteral);
        int rolesSize = in.readVInt();
        TreeSet<DiscoveryNodeRole> roles = new TreeSet<DiscoveryNodeRole>();
        String[] roleNames = new String[rolesSize];
        for (int i = 0; i < rolesSize; ++i) {
            String roleName = in.readString();
            String roleNameAbbreviation = in.readString();
            boolean canContainData = in.readBoolean();
            Optional<DiscoveryNodeRole> maybeRole = DiscoveryNodeRole.maybeGetRoleFromRoleName(roleName);
            if (maybeRole.isEmpty()) {
                roles.add(new DiscoveryNodeRole.UnknownRole(roleName, roleNameAbbreviation, canContainData));
                roleNames[i] = roleName;
                continue;
            }
            DiscoveryNodeRole role = maybeRole.get();
            assert (roleName.equals(role.roleName())) : "role name [" + roleName + "] does not match role [" + role.roleName() + "]";
            assert (roleNameAbbreviation.equals(role.roleNameAbbreviation())) : "role name abbreviation [" + roleName + "] does not match role [" + role.roleNameAbbreviation() + "]";
            roles.add(role);
            roleNames[i] = role.roleName();
        }
        this.roles = Collections.unmodifiableSortedSet(roles);
        if (in.getTransportVersion().onOrAfter(TransportVersions.V_8_10_X)) {
            Version version = Version.readVersion(in);
            IndexVersion minIndexVersion = IndexVersion.readVersion(in);
            IndexVersion minReadOnlyIndexVersion = in.getTransportVersion().supports(TransportVersions.V_8_18_0) ? IndexVersion.readVersion(in) : minIndexVersion;
            IndexVersion maxIndexVersion = IndexVersion.readVersion(in);
            this.versionInfo = new VersionInformation(version, minIndexVersion, minReadOnlyIndexVersion, maxIndexVersion);
        } else {
            this.versionInfo = DiscoveryNode.inferVersionInformation(Version.readVersion(in));
        }
        this.externalId = in.getTransportVersion().onOrAfter(EXTERNAL_ID_VERSION) ? readStringLiteral.read(in) : this.nodeName;
        this.roleNames = Set.of(roleNames);
    }

    public boolean hasRole(String roleName) {
        return this.roleNames.contains(roleName);
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        out.writeString(this.nodeName);
        out.writeString(this.nodeId);
        out.writeString(this.ephemeralId);
        out.writeString(this.hostName);
        out.writeString(this.hostAddress);
        this.address.writeTo(out);
        out.writeMap(this.attributes, StreamOutput::writeString);
        out.writeCollection(this.roles, (o, role) -> {
            o.writeString(role.roleName());
            o.writeString(role.roleNameAbbreviation());
            o.writeBoolean(role.canContainData());
        });
        if (out.getTransportVersion().onOrAfter(TransportVersions.V_8_10_X)) {
            Version.writeVersion(this.versionInfo.nodeVersion(), out);
            IndexVersion.writeVersion(this.versionInfo.minIndexVersion(), out);
            if (out.getTransportVersion().supports(TransportVersions.V_8_18_0)) {
                IndexVersion.writeVersion(this.versionInfo.minReadOnlyIndexVersion(), out);
            }
            IndexVersion.writeVersion(this.versionInfo.maxIndexVersion(), out);
        } else {
            Version.writeVersion(this.versionInfo.nodeVersion(), out);
        }
        if (out.getTransportVersion().onOrAfter(EXTERNAL_ID_VERSION)) {
            out.writeString(this.externalId);
        }
    }

    public TransportAddress getAddress() {
        return this.address;
    }

    public String getId() {
        return this.nodeId;
    }

    public String getEphemeralId() {
        return this.ephemeralId;
    }

    public String getName() {
        return this.nodeName;
    }

    public String getExternalId() {
        return this.externalId;
    }

    public Map<String, String> getAttributes() {
        return this.attributes;
    }

    public boolean canContainData() {
        return this.roles.stream().anyMatch(DiscoveryNodeRole::canContainData);
    }

    public boolean isMasterNode() {
        return this.roles.contains(DiscoveryNodeRole.MASTER_ROLE);
    }

    public boolean isIngestNode() {
        return this.roles.contains(DiscoveryNodeRole.INGEST_ROLE);
    }

    public boolean isRemoteClusterClient() {
        return this.roles.contains(DiscoveryNodeRole.REMOTE_CLUSTER_CLIENT_ROLE);
    }

    public boolean isDedicatedFrozenNode() {
        return DiscoveryNode.isDedicatedFrozenRoles(this.getRoles());
    }

    public Set<DiscoveryNodeRole> getRoles() {
        return this.roles;
    }

    public VersionInformation getVersionInformation() {
        return this.versionInfo;
    }

    public Version getVersion() {
        return this.versionInfo.nodeVersion();
    }

    public BuildVersion getBuildVersion() {
        return BuildVersion.fromVersionId(this.getVersion().id);
    }

    public OptionalInt getPre811VersionId() {
        int versionId = this.versionInfo.nodeVersion().id;
        if (versionId >= Version.V_8_11_0.id) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(versionId);
    }

    public IndexVersion getMinIndexVersion() {
        return this.versionInfo.minIndexVersion();
    }

    public IndexVersion getMinReadOnlyIndexVersion() {
        return this.versionInfo.minReadOnlyIndexVersion();
    }

    public IndexVersion getMaxIndexVersion() {
        return this.versionInfo.maxIndexVersion();
    }

    public String getHostName() {
        return this.hostName;
    }

    public String getHostAddress() {
        return this.hostAddress;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DiscoveryNode that = (DiscoveryNode)o;
        return this.ephemeralId.equals(that.ephemeralId);
    }

    public int hashCode() {
        return this.ephemeralId.hashCode();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        this.appendDescriptionWithoutAttributes(sb);
        if (!this.attributes.isEmpty()) {
            sb.append(this.attributes);
        }
        return sb.toString();
    }

    public void appendDescriptionWithoutAttributes(StringBuilder stringBuilder) {
        if (this.nodeName.length() > 0) {
            stringBuilder.append('{').append(this.nodeName).append('}');
        }
        stringBuilder.append('{').append(this.nodeId).append('}');
        stringBuilder.append('{').append(this.ephemeralId).append('}');
        if (this.externalId.length() > 0) {
            stringBuilder.append('{').append(this.externalId).append('}');
        }
        stringBuilder.append('{').append(this.hostName).append('}');
        stringBuilder.append('{').append(this.address).append('}');
        if (!this.roles.isEmpty()) {
            stringBuilder.append('{');
            this.appendRoleAbbreviations(stringBuilder, "");
            stringBuilder.append('}');
        }
        stringBuilder.append('{').append(this.versionInfo.nodeVersion()).append('}');
        stringBuilder.append('{').append(this.versionInfo.minIndexVersion()).append('-').append(this.versionInfo.maxIndexVersion()).append('}');
    }

    public void appendRoleAbbreviations(StringBuilder stringBuilder, String ifEmpty) {
        if (this.roles.isEmpty()) {
            stringBuilder.append(ifEmpty);
        } else {
            this.roles.stream().map(DiscoveryNodeRole::roleNameAbbreviation).sorted().forEach(stringBuilder::append);
        }
    }

    public String getRoleAbbreviationString() {
        StringBuilder stringBuilder = new StringBuilder();
        this.appendRoleAbbreviations(stringBuilder, "-");
        return stringBuilder.toString();
    }

    public String descriptionWithoutAttributes() {
        StringBuilder stringBuilder = new StringBuilder();
        this.appendDescriptionWithoutAttributes(stringBuilder);
        return stringBuilder.toString();
    }

    public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        builder.startObject(this.getId());
        builder.field("name", this.getName());
        builder.field("ephemeral_id", this.getEphemeralId());
        builder.field("transport_address", this.getAddress().toString());
        builder.field("external_id", this.getExternalId());
        builder.stringStringMap("attributes", this.attributes);
        builder.startArray("roles");
        for (DiscoveryNodeRole role : this.roles) {
            builder.value(role.roleName());
        }
        builder.endArray();
        builder.field("version", (ToXContent)this.versionInfo.nodeVersion());
        builder.field("min_index_version", (ToXContent)this.versionInfo.minIndexVersion());
        builder.field("max_index_version", (ToXContent)this.versionInfo.maxIndexVersion());
        builder.endObject();
        return builder;
    }

    public DiscoveryNode withTransportAddress(TransportAddress transportAddress) {
        return new DiscoveryNode(this.getName(), this.getId(), this.getEphemeralId(), this.getHostName(), this.getHostAddress(), transportAddress, this.getAttributes(), this.getRoles(), this.getVersionInformation(), this.getExternalId());
    }

    @Nullable
    public static String deduplicateNodeIdentifier(@Nullable String nodeIdentifier) {
        if (nodeIdentifier == null) {
            return null;
        }
        return nodeStringDeduplicator.deduplicate(nodeIdentifier);
    }
}

