"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleUnusedUrlsCleanupTask = exports.runDeleteUnusedUrlsTask = exports.getDeleteUnusedUrlTaskInstance = exports.fetchUnusedUrlsFromFirstNamespace = exports.durationToSeconds = exports.deleteUnusedUrls = void 0;
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const durationToSeconds = duration => `${duration.asSeconds()}s`;
exports.durationToSeconds = durationToSeconds;
const getDeleteUnusedUrlTaskInstance = interval => ({
  id: _constants.TASK_ID,
  taskType: _constants.TASK_ID,
  params: {},
  state: {},
  schedule: {
    interval: durationToSeconds(interval)
  },
  scope: ['share']
});
exports.getDeleteUnusedUrlTaskInstance = getDeleteUnusedUrlTaskInstance;
const deleteUnusedUrls = async ({
  savedObjectsRepository,
  unusedUrls,
  namespace,
  logger
}) => {
  try {
    logger.debug(`Deleting ${unusedUrls.length} unused URL(s) in namespace "${namespace}"`);
    await savedObjectsRepository.bulkDelete(unusedUrls, {
      refresh: 'wait_for',
      namespace
    });
    logger.debug(`Succesfully deleted ${unusedUrls.length} unused URL(s) in namespace "${namespace}"`);
  } catch (e) {
    throw new Error(`Failed to delete unused URL(s) in namespace "${namespace}": ${e.message}`);
  }
};
exports.deleteUnusedUrls = deleteUnusedUrls;
const fetchUnusedUrlsFromFirstNamespace = async ({
  savedObjectsRepository,
  urlExpirationDuration,
  urlLimit
}) => {
  var _savedObjects$, _savedObjects$$namesp;
  const filter = `url.attributes.accessDate <= now-${durationToSeconds(urlExpirationDuration)}`;
  const {
    saved_objects: savedObjects,
    total,
    per_page: perPage,
    page
  } = await savedObjectsRepository.find({
    type: _constants.SAVED_OBJECT_TYPE,
    filter,
    perPage: urlLimit,
    namespaces: ['*'],
    fields: ['type']
  });
  const firstNamespace = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString((_savedObjects$ = savedObjects[0]) === null || _savedObjects$ === void 0 ? void 0 : (_savedObjects$$namesp = _savedObjects$.namespaces) === null || _savedObjects$$namesp === void 0 ? void 0 : _savedObjects$$namesp[0]);
  const savedObjectsByNamespace = savedObjects.filter(so => {
    var _so$namespaces;
    return ((_so$namespaces = so.namespaces) === null || _so$namespaces === void 0 ? void 0 : _so$namespaces.length) && so.namespaces.includes(firstNamespace);
  });
  const unusedUrls = savedObjectsByNamespace.map(so => ({
    id: so.id,
    type: so.type
  }));
  return {
    unusedUrls,
    hasMore: page * perPage < total,
    namespace: firstNamespace
  };
};
exports.fetchUnusedUrlsFromFirstNamespace = fetchUnusedUrlsFromFirstNamespace;
const runDeleteUnusedUrlsTask = async ({
  core,
  urlExpirationDuration,
  urlLimit,
  logger,
  isEnabled
}) => {
  if (!isEnabled) {
    logger.debug('Unused URLs cleanup task is disabled, skipping execution');
    return {
      deletedCount: 0
    };
  }
  logger.debug('Unused URLs cleanup started');
  const [coreStart] = await core.getStartServices();
  const savedObjectsRepository = coreStart.savedObjects.createInternalRepository();
  let deletedCount = 0;
  let {
    unusedUrls,
    hasMore,
    namespace
  } = await fetchUnusedUrlsFromFirstNamespace({
    savedObjectsRepository,
    urlExpirationDuration,
    urlLimit
  });
  while (unusedUrls.length > 0 && deletedCount < urlLimit) {
    await deleteUnusedUrls({
      savedObjectsRepository,
      unusedUrls,
      logger,
      namespace
    });
    deletedCount += unusedUrls.length;
    if (hasMore && deletedCount < urlLimit) {
      const nextPage = await fetchUnusedUrlsFromFirstNamespace({
        savedObjectsRepository,
        urlExpirationDuration,
        urlLimit: urlLimit - deletedCount
      });
      unusedUrls = nextPage.unusedUrls;
      hasMore = nextPage.hasMore;
      namespace = nextPage.namespace;
    } else {
      break;
    }
  }
  logger.debug('Unused URLs cleanup finished');
  return {
    deletedCount
  };
};
exports.runDeleteUnusedUrlsTask = runDeleteUnusedUrlsTask;
const scheduleUnusedUrlsCleanupTask = async ({
  taskManager,
  checkInterval,
  isEnabled
}) => {
  try {
    if (!isEnabled) {
      await taskManager.removeIfExists(_constants.TASK_ID);
      return;
    }
    const taskInstance = getDeleteUnusedUrlTaskInstance(checkInterval);
    await taskManager.ensureScheduled(taskInstance);
    // TODO: Workaround for interval not updating on change until https://github.com/elastic/kibana/issues/222089 is resolved.
    await taskManager.bulkUpdateSchedules([_constants.TASK_ID], {
      interval: durationToSeconds(checkInterval)
    });
  } catch (e) {
    throw new Error(e.message || 'Failed to schedule unused URLs cleanup task');
  }
};
exports.scheduleUnusedUrlsCleanupTask = scheduleUnusedUrlsCleanupTask;