"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformFindAlerts = exports.transformAlertsToRules = exports.transform = exports.swapActionIds = exports.separateActionsAndSystemAction = exports.migrateLegacyInvestigationFields = exports.migrateLegacyActionsIds = exports.getTupleDuplicateErrorsAndUniqueRules = exports.getIdError = exports.getIdBulkError = exports.createBulkActionError = void 0;
var _fp = require("lodash/fp");
var _pMap = _interopRequireDefault(require("p-map"));
var _uuid = require("uuid");
var _utils = require("../../routes/utils");
var _rule_schema = require("../../rule_schema");
var _internal_rule_to_api_response = require("../logic/detection_rules_client/converters/internal_rule_to_api_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CONCURRENT_SEARCHES = 10;
const getIdError = ({
  id,
  ruleId
}) => {
  if (id != null) {
    return {
      message: `id: "${id}" not found`,
      statusCode: 404
    };
  } else if (ruleId != null) {
    return {
      message: `rule_id: "${ruleId}" not found`,
      statusCode: 404
    };
  } else {
    return {
      message: 'id or rule_id should have been defined',
      statusCode: 404
    };
  }
};
exports.getIdError = getIdError;
const getIdBulkError = ({
  id,
  ruleId
}) => {
  if (id != null && ruleId != null) {
    return (0, _utils.createBulkErrorObject)({
      id,
      ruleId,
      statusCode: 404,
      message: `id: "${id}" and rule_id: "${ruleId}" not found`
    });
  } else if (id != null) {
    return (0, _utils.createBulkErrorObject)({
      id,
      statusCode: 404,
      message: `id: "${id}" not found`
    });
  } else if (ruleId != null) {
    return (0, _utils.createBulkErrorObject)({
      ruleId,
      statusCode: 404,
      message: `rule_id: "${ruleId}" not found`
    });
  } else {
    return (0, _utils.createBulkErrorObject)({
      statusCode: 404,
      message: `id or rule_id should have been defined`
    });
  }
};
exports.getIdBulkError = getIdBulkError;
const transformAlertsToRules = rules => {
  return rules.map(rule => (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule));
};
exports.transformAlertsToRules = transformAlertsToRules;
const transformFindAlerts = ruleFindResults => {
  return {
    page: ruleFindResults.page,
    perPage: ruleFindResults.perPage,
    total: ruleFindResults.total,
    data: ruleFindResults.data.map(rule => {
      return (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule);
    })
  };
};
exports.transformFindAlerts = transformFindAlerts;
const transform = rule => {
  if ((0, _rule_schema.hasValidRuleType)(rule)) {
    return (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule);
  }
  return null;
};
exports.transform = transform;
const getTupleDuplicateErrorsAndUniqueRules = (rules, isOverwrite) => {
  const {
    errors,
    rulesAcc
  } = rules.reduce((acc, parsedRule) => {
    if (parsedRule instanceof Error) {
      acc.rulesAcc.set((0, _uuid.v4)(), parsedRule);
    } else {
      const {
        rule_id: ruleId
      } = parsedRule;
      if (acc.rulesAcc.has(ruleId) && !isOverwrite) {
        acc.errors.set((0, _uuid.v4)(), (0, _utils.createBulkErrorObject)({
          ruleId,
          statusCode: 400,
          message: `More than one rule with rule-id: "${ruleId}" found`
        }));
      }
      acc.rulesAcc.set(ruleId, parsedRule);
    }
    return acc;
  },
  // using map (preserves ordering)
  {
    errors: new Map(),
    rulesAcc: new Map()
  });
  return [Array.from(errors.values()), Array.from(rulesAcc.values())];
};

// functions copied from here
// https://github.com/elastic/kibana/blob/4584a8b570402aa07832cf3e5b520e5d2cfa7166/src/core/server/saved_objects/import/lib/check_origin_conflicts.ts#L55-L57
exports.getTupleDuplicateErrorsAndUniqueRules = getTupleDuplicateErrorsAndUniqueRules;
const createQueryTerm = input => input.replace(/\\/g, '\\\\').replace(/\"/g, '\\"');
const createQuery = (type, id) => `"${createQueryTerm(`${type}:${id}`)}" | "${createQueryTerm(id)}"`;

/**
 * Query for a saved object with a given origin id and replace the
 * id in the provided action with the _id from the query result
 * @param action
 * @param esClient
 * @returns
 */
const swapActionIds = async (action, savedObjectsClient) => {
  try {
    const search = createQuery('action', action.id);
    const foundAction = await savedObjectsClient.find({
      type: 'action',
      search,
      rootSearchFields: ['_id', 'originId']
    });
    if (foundAction.saved_objects.length === 1) {
      return {
        ...action,
        id: foundAction.saved_objects[0].id
      };
    } else if (foundAction.saved_objects.length > 1) {
      return new Error(`Found two action connectors with originId or _id: ${action.id} The upload cannot be completed unless the _id or the originId of the action connector is changed. See https://www.elastic.co/guide/en/kibana/current/sharing-saved-objects.html for more details`);
    }
    return action;
  } catch (exc) {
    return exc;
  }
};

/**
 * In 8.0 all saved objects made in a non-default space will have their
 * _id's regenerated. Security Solution rules have references to the
 * actions SO id inside the 'actions' param.
 * When users import these rules, we need to ensure any rule with
 * an action that has an old, pre-8.0 id will need to be updated
 * to point to the new _id for that action (alias_target_id)
 *
 * ex:
 * import rule.ndjson:
 * {
 *   rule_id: 'myrule_id'
 *   name: 'my favorite rule'
 *   ...
 *   actions:[{id: '1111-2222-3333-4444', group...}]
 * }
 *
 * In 8.0 the 'id' field of this action is no longer a reference
 * to the _id of the action (connector). Querying against the connector
 * endpoint for this id will yield 0 results / 404.
 *
 * The solution: If we query the .kibana index for '1111-2222-3333-4444' as an originId,
 * we should get the original connector back
 * (with the new, migrated 8.0 _id of 'xxxx-yyyy-zzzz-0000') and can then replace
 * '1111-2222-3333-4444' in the example above with 'xxxx-yyyy-zzzz-0000'
 * And the rule will then import successfully.
 * @param rules
 * @param savedObjectsClient SO client exposing hidden 'actions' SO type
 * @returns
 */
exports.swapActionIds = swapActionIds;
const migrateLegacyActionsIds = async (rules, savedObjectsClient, actionsClient) => {
  const isImportRule = r => !(r instanceof Error);
  const toReturn = await (0, _pMap.default)(rules, async rule => {
    if (isImportRule(rule) && rule.actions != null && !(0, _fp.isEmpty)(rule.actions)) {
      var _ref;
      // filter out system actions, since they were not part of any 7.x releases and do not need to be migrated
      const [systemActions, extActions] = (0, _fp.partition)(action => actionsClient.isSystemAction(action.id))(rule.actions);
      // can we swap the pre 8.0 action connector(s) id with the new,
      // post-8.0 action id (swap the originId for the new _id?)
      const newActions = await (0, _pMap.default)((_ref = extActions) !== null && _ref !== void 0 ? _ref : [], action => swapActionIds(action, savedObjectsClient), {
        concurrency: MAX_CONCURRENT_SEARCHES
      });

      // were there any errors discovered while trying to migrate and swap the action connector ids?
      const [actionMigrationErrors, newlyMigratedActions] = (0, _fp.partition)(item => item instanceof Error)(newActions);
      if (actionMigrationErrors == null || actionMigrationErrors.length === 0) {
        return {
          ...rule,
          actions: [...newlyMigratedActions, ...systemActions]
        };
      }
      return [{
        ...rule,
        actions: [...newlyMigratedActions, ...systemActions]
      }, new Error(JSON.stringify((0, _utils.createBulkErrorObject)({
        ruleId: rule.rule_id,
        statusCode: 409,
        message: `${actionMigrationErrors.map(error => error.message).join(',')}`
      })))];
    }
    return rule;
  }, {
    concurrency: MAX_CONCURRENT_SEARCHES
  });
  return toReturn.flat();
};

/**
 * In ESS 8.10.x "investigation_fields" are mapped as string[].
 * For 8.11+ logic is added on read in our endpoints to migrate
 * the data over to it's intended type of { field_names: string[] }.
 * The SO rule type will continue to support both types until we deprecate,
 * but APIs will only support intended object format.
 * See PR 169061
 */
exports.migrateLegacyActionsIds = migrateLegacyActionsIds;
const migrateLegacyInvestigationFields = investigationFields => {
  if (investigationFields && Array.isArray(investigationFields)) {
    if (investigationFields.length) {
      return {
        field_names: investigationFields
      };
    }
    return undefined;
  }
  return investigationFields;
};
exports.migrateLegacyInvestigationFields = migrateLegacyInvestigationFields;
const separateActionsAndSystemAction = (actionsClient, actions) => !(0, _fp.isEmpty)(actions) ? (0, _fp.partition)(action => actionsClient.isSystemAction(action.id))(actions) : [[], actions];
exports.separateActionsAndSystemAction = separateActionsAndSystemAction;
const createBulkActionError = ({
  message,
  statusCode,
  id
}) => {
  const error = new Error(message);
  error.statusCode = statusCode;
  return {
    item: id,
    error
  };
};
exports.createBulkActionError = createBulkActionError;