"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSoFindIterable = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/no-explicit-any */

/**
 * Creates an `AsyncIterable` that can be used to iterate (ex. via `for..await..of`) over all the data
 * matching the search query. The search request to Saved Object will use `searchAfter`, thus can iterate over
 * datasets above 10k items as well.
 *
 * @param options
 */
const createSoFindIterable = ({
  soClient,
  findRequest: {
    perPage = 1000,
    ...findOptions
  },
  resultsMapper,
  usePointInTime = true,
  namespaces = ['*']
}) => {
  const keepAliveValue = '5m';
  let done = false;
  let value;
  let searchAfterValue;
  let pointInTime = usePointInTime ? soClient.openPointInTimeForType(findOptions.type, {
    keepAlive: keepAliveValue
  }) : Promise.resolve({
    id: ''
  });
  const setValue = findResponse => {
    value = resultsMapper ? resultsMapper(findResponse) : findResponse;
  };
  const setDone = async () => {
    done = true;
    if (usePointInTime) {
      const pitId = (await pointInTime).id;
      if (pitId) {
        await soClient.closePointInTime(pitId);
      }
    }
  };
  const fetchData = async () => {
    var _findResult$pit_id;
    const findResult = await soClient.find({
      ...findOptions,
      ...(usePointInTime ? {
        pit: {
          id: (await pointInTime).id,
          keepAlive: keepAliveValue
        }
      } : {}),
      perPage,
      searchAfter: searchAfterValue,
      namespaces
    }).catch(e => {
      Error.captureStackTrace(e);
      throw e;
    });
    const soItems = findResult.saved_objects;
    const lastSearchHit = soItems[soItems.length - 1];
    if (soItems.length === 0) {
      setValue(findResult);
      await setDone();
      return;
    }

    // eslint-disable-next-line require-atomic-updates
    searchAfterValue = lastSearchHit.sort;
    // eslint-disable-next-line require-atomic-updates
    pointInTime = Promise.resolve({
      id: (_findResult$pit_id = findResult.pit_id) !== null && _findResult$pit_id !== void 0 ? _findResult$pit_id : ''
    });
    setValue(findResult);

    // If (for some reason) we don't have a `searchAfterValue`,
    // then throw an error, or else we'll keep looping forever
    if (!searchAfterValue) {
      await setDone();
      throw new Error(`Unable to store 'searchAfter' value. Last 'SavedObjectsFindResult' did not include a 'sort' property \n(did you forget to set the 'sortField' attribute on your SavedObjectsFindOptions?)':\n${JSON.stringify(lastSearchHit)}`);
    }
  };
  const createIteratorResult = () => {
    return {
      done,
      value
    };
  };
  return {
    [Symbol.asyncIterator]() {
      return {
        async next() {
          if (!done) {
            await fetchData();
          }
          return createIteratorResult();
        },
        async return() {
          done = true;
          return createIteratorResult();
        }
      };
    }
  };
};
exports.createSoFindIterable = createSoFindIterable;