"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MicrosoftDefenderEndpointActionsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/stack-connectors-plugin/common/microsoft_defender_endpoint/constants");
var _lodash = require("lodash");
var _base_response_actions_client = require("../../../lib/base_response_actions_client");
var _stringify = require("../../../../../../utils/stringify");
var _errors = require("../../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class MicrosoftDefenderEndpointActionsClient extends _base_response_actions_client.ResponseActionsClientImpl {
  constructor({
    connectorActions,
    ...options
  }) {
    super(options);
    (0, _defineProperty2.default)(this, "agentType", 'microsoft_defender_endpoint');
    (0, _defineProperty2.default)(this, "connectorActionsClient", void 0);
    this.connectorActionsClient = connectorActions;
    connectorActions.setup(_constants.MICROSOFT_DEFENDER_ENDPOINT_CONNECTOR_ID);
  }
  async handleResponseActionCreation(actionRequestOptions) {
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(actionRequestOptions);
    await this.updateCases({
      command: actionRequestOptions.command,
      caseIds: actionRequestOptions.case_ids,
      alertIds: actionRequestOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: actionRequestOptions.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End, _actionRequestDoc$End2;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End = (_actionRequestDoc$End2 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End2 === void 0 ? void 0 : _actionRequestDoc$End2[agentId].name) !== null && _actionRequestDoc$End !== void 0 ? _actionRequestDoc$End : ''
        };
      }),
      comment: actionRequestOptions.comment
    });
    return {
      actionEsDoc: actionRequestDoc,
      actionDetails: await this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id)
    };
  }

  /**
   * Sends actions to Ms Defender for Endpoint directly (via Connector)
   * @internal
   */
  async sendAction(actionType, actionParams) {
    const executeOptions = {
      params: {
        subAction: actionType,
        subActionParams: actionParams
      }
    };
    this.log.debug(() => `calling connector actions 'execute()' for Microsoft Defender for Endpoint with:\n${(0, _stringify.stringify)(executeOptions)}`);
    const actionSendResponse = await this.connectorActionsClient.execute(executeOptions);
    if (actionSendResponse.status === 'error') {
      this.log.error((0, _stringify.stringify)(actionSendResponse));
      throw new _errors.ResponseActionsClientError(`Attempt to send [${actionType}] to Microsoft Defender for Endpoint failed: ${actionSendResponse.serviceMessage || actionSendResponse.message}`, 500, actionSendResponse);
    }
    this.log.debug(() => `Response:\n${(0, _stringify.stringify)(actionSendResponse)}`);
    return actionSendResponse;
  }
  async writeActionRequestToEndpointIndex(actionRequest) {
    const agentId = actionRequest.endpoint_ids[0];
    const agentDetails = await this.getAgentDetails(agentId);
    const doc = await super.writeActionRequestToEndpointIndex({
      ...actionRequest,
      hosts: {
        [agentId]: {
          name: agentDetails.computerDnsName
        }
      }
    });
    return doc;
  }

  /** Gets agent details directly from MS Defender for Endpoint */
  async getAgentDetails(agentId) {
    const cachedEntry = this.cache.get(agentId);
    if (cachedEntry) {
      this.log.debug(`Found cached agent details for UUID [${agentId}]:\n${(0, _stringify.stringify)(cachedEntry)}`);
      return cachedEntry;
    }
    let msDefenderEndpointGetMachineDetailsApiResponse;
    try {
      const agentDetailsResponse = await this.sendAction(_constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_AGENT_DETAILS, {
        id: agentId
      });
      msDefenderEndpointGetMachineDetailsApiResponse = agentDetailsResponse.data;
    } catch (err) {
      throw new _errors.ResponseActionsClientError(`Error while attempting to retrieve Microsoft Defender for Endpoint host with agent id [${agentId}]: ${err.message}`, 500, err);
    }
    if (!msDefenderEndpointGetMachineDetailsApiResponse) {
      throw new _errors.ResponseActionsClientError(`Microsoft Defender for Endpoint agent id [${agentId}] not found`, 404);
    }
    this.cache.set(agentId, msDefenderEndpointGetMachineDetailsApiResponse);
    return msDefenderEndpointGetMachineDetailsApiResponse;
  }
  async validateRequest(payload) {
    // TODO: support multiple agents
    if (payload.endpoint_ids.length > 1) {
      return {
        isValid: false,
        error: new _errors.ResponseActionsClientError(`[body.endpoint_ids]: Multiple agents IDs not currently supported for Microsoft Defender for Endpoint`, 400)
      };
    }
    return super.validateRequest(payload);
  }
  async isolate(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'isolate'
    };
    if (!reqIndexOptions.error) {
      var _error;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          var _msActionResponse$dat;
          const msActionResponse = await this.sendAction(_constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.ISOLATE_HOST, {
            id: actionRequest.endpoint_ids[0],
            comment: this.buildExternalComment(reqIndexOptions)
          });
          if (msActionResponse !== null && msActionResponse !== void 0 && (_msActionResponse$dat = msActionResponse.data) !== null && _msActionResponse$dat !== void 0 && _msActionResponse$dat.id) {
            reqIndexOptions.meta = {
              machineActionId: msActionResponse.data.id
            };
          } else {
            throw new _errors.ResponseActionsClientError(`Isolate request was sent to Microsoft Defender, but Machine Action Id was not provided!`);
          }
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error = error) === null || _error === void 0 ? void 0 : _error.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async release(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'unisolate'
    };
    if (!reqIndexOptions.error) {
      var _error2;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          var _msActionResponse$dat2;
          const msActionResponse = await this.sendAction(_constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.RELEASE_HOST, {
            id: actionRequest.endpoint_ids[0],
            comment: this.buildExternalComment(reqIndexOptions)
          });
          if (msActionResponse !== null && msActionResponse !== void 0 && (_msActionResponse$dat2 = msActionResponse.data) !== null && _msActionResponse$dat2 !== void 0 && _msActionResponse$dat2.id) {
            reqIndexOptions.meta = {
              machineActionId: msActionResponse.data.id
            };
          } else {
            throw new _errors.ResponseActionsClientError(`Un-Isolate request was sent to Microsoft Defender, but Machine Action Id was not provided!`);
          }
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error2 = error) === null || _error2 === void 0 ? void 0 : _error2.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async runscript(actionRequest, options) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'runscript'
    };
    if (!reqIndexOptions.error) {
      var _error3;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          var _msActionResponse$dat3;
          const msActionResponse = await this.sendAction(_constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.RUN_SCRIPT, {
            id: reqIndexOptions.endpoint_ids[0],
            comment: this.buildExternalComment(reqIndexOptions),
            parameters: {
              scriptName: reqIndexOptions.parameters.scriptName,
              args: reqIndexOptions.parameters.args
            }
          });
          if (msActionResponse !== null && msActionResponse !== void 0 && (_msActionResponse$dat3 = msActionResponse.data) !== null && _msActionResponse$dat3 !== void 0 && _msActionResponse$dat3.id) {
            reqIndexOptions.meta = {
              machineActionId: msActionResponse.data.id
            };
          } else {
            throw new _errors.ResponseActionsClientError(`Run Script request was sent to Microsoft Defender, but Machine Action Id was not provided!`);
          }
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error3 = error) === null || _error3 === void 0 ? void 0 : _error3.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async processPendingActions({
    abortSignal,
    addToQueue
  }) {
    if (abortSignal.aborted) {
      return;
    }
    const addResponsesToQueueIfAny = responseList => {
      if (responseList.length > 0) {
        addToQueue(...responseList);
        this.sendActionResponseTelemetry(responseList);
      }
    };
    for await (const pendingActions of this.fetchAllPendingActions()) {
      if (abortSignal.aborted) {
        return;
      }
      const pendingActionsByType = (0, _lodash.groupBy)(pendingActions, 'action.EndpointActions.data.command');
      for (const [actionType, typePendingActions] of Object.entries(pendingActionsByType)) {
        if (abortSignal.aborted) {
          return;
        }
        switch (actionType) {
          case 'isolate':
          case 'unisolate':
            addResponsesToQueueIfAny(await this.checkPendingActions(typePendingActions));
          case 'runscript':
            addResponsesToQueueIfAny(await this.checkPendingActions(typePendingActions, {
              downloadResult: true
            }));
        }
      }
    }
  }
  async checkPendingActions(actionRequests, options = {
    downloadResult: false
  }) {
    const completedResponses = [];
    const warnings = [];
    const actionsByMachineId = {};
    const machineActionIds = [];
    const msApiOptions = {
      id: machineActionIds,
      pageSize: 1000
    };
    for (const {
      action
    } of actionRequests) {
      var _action$meta;
      const command = action.EndpointActions.data.command;
      const machineActionId = (_action$meta = action.meta) === null || _action$meta === void 0 ? void 0 : _action$meta.machineActionId;
      if (!machineActionId) {
        warnings.push(`${command} response action ID [${action.EndpointActions.action_id}] is missing Microsoft Defender for Endpoint machine action id, thus unable to check on it's status. Forcing it to complete as failure.`);
        completedResponses.push(this.buildActionResponseEsDoc({
          actionId: action.EndpointActions.action_id,
          agentId: Array.isArray(action.agent.id) ? action.agent.id[0] : action.agent.id,
          data: {
            command
          },
          error: {
            message: `Unable to verify if action completed. Microsoft Defender machine action id ('meta.machineActionId') missing from the action request document!`
          }
        }));
      } else {
        if (!actionsByMachineId[machineActionId]) {
          actionsByMachineId[machineActionId] = [];
        }
        actionsByMachineId[machineActionId].push(action);
        machineActionIds.push(machineActionId);
      }
    }
    const {
      data: machineActions
    } = await this.sendAction(_constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_ACTIONS, msApiOptions);
    if (machineActions !== null && machineActions !== void 0 && machineActions.value) {
      for (const machineAction of machineActions.value) {
        var _machineAction$comman, _machineAction$comman2, _machineAction$comman3, _machineAction$comman4;
        const {
          isPending,
          isError,
          message
        } = this.calculateMachineActionState(machineAction);
        const commandErrors = (_machineAction$comman = (_machineAction$comman2 = machineAction.commands) === null || _machineAction$comman2 === void 0 ? void 0 : (_machineAction$comman3 = _machineAction$comman2[0]) === null || _machineAction$comman3 === void 0 ? void 0 : (_machineAction$comman4 = _machineAction$comman3.errors) === null || _machineAction$comman4 === void 0 ? void 0 : _machineAction$comman4.join('\n')) !== null && _machineAction$comman !== void 0 ? _machineAction$comman : '';
        if (!isPending) {
          var _actionsByMachineId$m;
          const pendingActionRequests = (_actionsByMachineId$m = actionsByMachineId[machineAction.id]) !== null && _actionsByMachineId$m !== void 0 ? _actionsByMachineId$m : [];
          for (const actionRequest of pendingActionRequests) {
            let additionalData = {};
            // In order to not copy paste most of the logic, I decided to add this additional check here to support `runscript` action and it's result that comes back as a link to download the file
            if (options.downloadResult) {
              additionalData = {
                meta: {
                  machineActionId: machineAction.id,
                  filename: `runscript-output-${machineAction.id}.json`,
                  createdAt: new Date().toISOString()
                }
              };
            }
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId: actionRequest.EndpointActions.action_id,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command: actionRequest.EndpointActions.data.command
              },
              error: isError ? {
                message: commandErrors || message
              } : undefined,
              ...additionalData
            }));
          }
        }
      }
    }
    this.log.debug(() => `${completedResponses.length} action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }
  calculateMachineActionState(machineAction) {
    let isPending = true;
    let isError = false;
    let message = '';
    switch (machineAction.status) {
      case 'Failed':
      case 'TimeOut':
        isPending = false;
        isError = true;
        message = `Response action ${machineAction.status} (Microsoft Defender for Endpoint machine action ID: ${machineAction.id})`;
        break;
      case 'Cancelled':
        isPending = false;
        isError = true;
        message = `Response action was canceled by [${machineAction.cancellationRequestor}] (Microsoft Defender for Endpoint machine action ID: ${machineAction.id})${machineAction.cancellationComment ? `: ${machineAction.cancellationComment}` : ''}`;
        break;
      case 'Succeeded':
        isPending = false;
        isError = false;
        break;
      default:
        // covers 'Pending' | 'InProgress'
        isPending = true;
        isError = false;
    }
    return {
      isPending,
      isError,
      message
    };
  }
  async getCustomScripts() {
    try {
      var _customScriptsRespons;
      const customScriptsResponse = await this.sendAction(_constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_LIBRARY_FILES, {});
      const scripts = ((_customScriptsRespons = customScriptsResponse.data) === null || _customScriptsRespons === void 0 ? void 0 : _customScriptsRespons.value) || [];

      // Transform MS Defender scripts to CustomScriptsResponse format
      const data = scripts.map(script => ({
        // due to External EDR's schema nature - we expect a maybe() everywhere - empty strings are needed
        id: script.fileName || '',
        name: script.fileName || '',
        description: script.description || ''
      }));
      return {
        data
      };
    } catch (err) {
      const error = new _errors.ResponseActionsClientError(`Failed to fetch Microsoft Defender for Endpoint scripts, failed with: ${err.message}`, 500, err);
      this.log.error(error);
      throw error;
    }
  }
  async getFileInfo(actionId, agentId) {
    await this.ensureValidActionId(actionId);
    const {
      EndpointActions: {
        data: {
          command
        }
      }
    } = await this.fetchActionRequestEsDoc(actionId);
    const {
      microsoftDefenderEndpointRunScriptEnabled
    } = this.options.endpointService.experimentalFeatures;
    if (command === 'runscript' && !microsoftDefenderEndpointRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`File downloads are not supported for ${this.agentType} agent type. Feature disabled.`, 400);
    }
    const fileInfo = {
      actionId,
      agentId,
      id: agentId,
      agentType: this.agentType,
      status: 'AWAITING_UPLOAD',
      created: '',
      name: '',
      size: 0,
      mimeType: ''
    };
    try {
      switch (command) {
        case 'runscript':
          {
            var _agentResponse$meta$c, _agentResponse$meta, _agentResponse$meta$f, _agentResponse$meta2;
            const agentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);
            fileInfo.status = 'READY';
            fileInfo.created = (_agentResponse$meta$c = (_agentResponse$meta = agentResponse.meta) === null || _agentResponse$meta === void 0 ? void 0 : _agentResponse$meta.createdAt) !== null && _agentResponse$meta$c !== void 0 ? _agentResponse$meta$c : '';
            fileInfo.name = (_agentResponse$meta$f = (_agentResponse$meta2 = agentResponse.meta) === null || _agentResponse$meta2 === void 0 ? void 0 : _agentResponse$meta2.filename) !== null && _agentResponse$meta$f !== void 0 ? _agentResponse$meta$f : '';
            fileInfo.mimeType = 'application/octet-stream';
          }
          break;
        default:
          throw new _errors.ResponseActionsClientError(`${command} does not support file downloads`, 400);
      }
    } catch (e) {
      // Ignore "no response doc" error for the agent and just return the file info with the status of 'AWAITING_UPLOAD'
      if (!(e instanceof _errors.ResponseActionAgentResponseEsDocNotFound)) {
        throw e;
      }
    }
    return fileInfo;
  }
  async getFileDownload(actionId, agentId) {
    await this.ensureValidActionId(actionId);
    const {
      EndpointActions: {
        data: {
          command
        }
      }
    } = await this.fetchActionRequestEsDoc(actionId);
    const {
      microsoftDefenderEndpointRunScriptEnabled
    } = this.options.endpointService.experimentalFeatures;
    if (command === 'runscript' && !microsoftDefenderEndpointRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`File downloads are not supported for ${this.agentType} agent type. Feature disabled.`, 400);
    }
    let downloadStream;
    let fileName = 'download.json';
    try {
      switch (command) {
        case 'runscript':
          {
            var _runscriptAgentRespon, _runscriptAgentRespon2;
            const runscriptAgentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);
            if (!((_runscriptAgentRespon = runscriptAgentResponse.meta) !== null && _runscriptAgentRespon !== void 0 && _runscriptAgentRespon.machineActionId)) {
              throw new _errors.ResponseActionsClientError(`Unable to retrieve file from Microsoft Defender for Endpoint. Response ES document is missing [meta.machineActionId]`);
            }
            const {
              data
            } = await this.sendAction(_constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_ACTION_RESULTS, {
              id: (_runscriptAgentRespon2 = runscriptAgentResponse.meta) === null || _runscriptAgentRespon2 === void 0 ? void 0 : _runscriptAgentRespon2.machineActionId
            });
            if (data) {
              downloadStream = data;
              fileName = runscriptAgentResponse.meta.filename;
            }
          }
          break;
      }
      if (!downloadStream) {
        throw new _errors.ResponseActionsClientError(`Unable to establish a file download Readable stream with Microsoft Defender for Endpoint for response action [${command}] [${actionId}]`);
      }
    } catch (e) {
      this.log.debug(() => `Attempt to get file download stream from Microsoft Defender for Endpoint for response action failed with:\n${(0, _stringify.stringify)(e)}`);
      throw e;
    }
    return {
      stream: downloadStream,
      mimeType: undefined,
      fileName
    };
  }
  async fetchEsResponseDocForAgentId(actionId, agentId) {
    const agentResponse = (await this.fetchActionResponseEsDocs(actionId, [agentId]))[agentId];
    if (!agentResponse) {
      throw new _errors.ResponseActionAgentResponseEsDocNotFound(`Action ID [${actionId}] for agent ID [${agentId}] is still pending`, 404);
    }
    return agentResponse;
  }
}
exports.MicrosoftDefenderEndpointActionsClient = MicrosoftDefenderEndpointActionsClient;