"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointActionsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _custom_http_request_error = require("../../../../../utils/custom_http_request_error");
var _utils = require("../../utils");
var _errors = require("../errors");
var _stringify = require("../../../../utils/stringify");
var _base_response_actions_client = require("../lib/base_response_actions_client");
var _constants = require("../../../../../../common/endpoint/service/response_actions/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getInvalidAgentsWarning = invalidAgents => invalidAgents.length ? `The following agent ids are not valid: ${JSON.stringify(invalidAgents)} and will not be included in action request` : '';
class EndpointActionsClient extends _base_response_actions_client.ResponseActionsClientImpl {
  constructor(...args) {
    super(...args);
    (0, _defineProperty2.default)(this, "agentType", 'endpoint');
  }
  async checkAgentIds(ids) {
    const uniqueIds = [...new Set(ids)];
    const foundEndpointHosts = await this.options.endpointService.getEndpointMetadataService().getMetadataForEndpoints(uniqueIds);
    const validIds = foundEndpointHosts.map(endpoint => endpoint.elastic.agent.id);
    const invalidIds = ids.filter(id => !validIds.includes(id));
    if (invalidIds.length) {
      this.log.warn(getInvalidAgentsWarning(invalidIds));
    }
    return {
      valid: validIds,
      invalid: invalidIds,
      allValid: invalidIds.length === 0,
      hosts: foundEndpointHosts
    };
  }
  async handleResponseAction(command, actionReq, options) {
    const validatedAgents = await this.checkAgentIds(actionReq.endpoint_ids);
    const actionId = (0, _uuid.v4)();
    const {
      error: validationError
    } = await this.validateRequest({
      ...actionReq,
      command,
      endpoint_ids: validatedAgents.valid || []
    });
    const {
      hosts,
      ruleName,
      ruleId,
      error
    } = this.getMethodOptions(options);
    let actionError = (validationError === null || validationError === void 0 ? void 0 : validationError.message) || error;
    if (actionError && !this.options.isAutomated) {
      throw new _errors.ResponseActionsClientError(actionError, 400);
    }

    // Dispatch action to Endpoint using Fleet
    if (!actionError) {
      try {
        await this.dispatchActionViaFleet({
          actionId,
          agents: validatedAgents.valid,
          data: {
            command,
            comment: actionReq.comment,
            parameters: actionReq.parameters
          }
        });
      } catch (e) {
        // If not in Automated mode, then just throw, else save the error and write
        // it to the Endpoint Action request doc
        if (!this.options.isAutomated) {
          throw e;
        }
        actionError = e.message;
      }
    }

    // Append warning message to comment if there are invalid agents
    const commentMessage = actionReq.comment ? actionReq.comment : '';
    const warningMessage = `(WARNING: ${getInvalidAgentsWarning(validatedAgents.invalid)})`;
    const comment = validatedAgents.invalid.length ? commentMessage ? `${commentMessage}. ${warningMessage}` : warningMessage : actionReq.comment;

    // Write action to endpoint index
    await this.writeActionRequestToEndpointIndex({
      ...actionReq,
      endpoint_ids: validatedAgents.valid,
      error: actionError,
      ruleId,
      ruleName,
      hosts,
      actionId,
      command,
      comment
    });

    // Update cases
    await this.updateCases({
      command,
      actionId,
      comment,
      caseIds: actionReq.case_ids,
      alertIds: actionReq.alert_ids,
      hosts: validatedAgents.valid.map(hostId => {
        var _ref, _validatedAgents$host, _validatedAgents$host2;
        return {
          hostId,
          hostname: (_ref = (_validatedAgents$host = (_validatedAgents$host2 = validatedAgents.hosts.find(host => host.agent.id === hostId)) === null || _validatedAgents$host2 === void 0 ? void 0 : _validatedAgents$host2.host.hostname) !== null && _validatedAgents$host !== void 0 ? _validatedAgents$host : hosts === null || hosts === void 0 ? void 0 : hosts[hostId].name) !== null && _ref !== void 0 ? _ref : ''
        };
      })
    });
    return this.fetchActionDetails(actionId);
  }
  async dispatchActionViaFleet({
    actionId = (0, _uuid.v4)(),
    agents,
    data
  }) {
    const [messageSigningService, fleetActionsService] = await Promise.all([this.options.endpointService.getMessageSigningService(), this.options.endpointService.getFleetActionsClient()]);
    const fleetActionDoc = {
      '@timestamp': new Date().toISOString(),
      input_type: 'endpoint',
      type: 'INPUT_ACTION',
      timeout: 300,
      // 5 minutes
      expiration: (0, _utils.getActionRequestExpiration)(),
      user_id: this.options.username,
      action_id: actionId || (0, _uuid.v4)(),
      agents,
      // Casting needed because Fleet type definition, which accepts any object, but was not defined using `any`
      data: data
    };
    const signedAction = await messageSigningService.sign(fleetActionDoc);
    fleetActionDoc.signed = {
      data: signedAction.data.toString('base64'),
      signature: signedAction.signature
    };
    this.log.debug(`Signed Fleet endpoint action request:\n${(0, _stringify.stringify)(fleetActionDoc)}`);
    return fleetActionsService.create(fleetActionDoc).catch(err => {
      const error = new _errors.ResponseActionsClientError(`Attempt to create Fleet [${data.command}] response action request for agents [${agents.join(', ')}] failed with: ${err.message}`, 500, err);
      this.log.error(error);
      throw error;
    });
  }
  async isolate(actionRequest, options = {}) {
    return this.handleResponseAction('isolate', actionRequest, options);
  }
  async release(actionRequest, options = {}) {
    return this.handleResponseAction('unisolate', actionRequest, options);
  }
  async killProcess(actionRequest, options = {}) {
    return this.handleResponseAction('kill-process', actionRequest, options);
  }
  async suspendProcess(actionRequest, options = {}) {
    return this.handleResponseAction('suspend-process', actionRequest, options);
  }
  async runningProcesses(actionRequest, options = {}) {
    return this.handleResponseAction('running-processes', actionRequest, options);
  }
  async getFile(actionRequest, options = {}) {
    return this.handleResponseAction('get-file', actionRequest, options);
  }
  async execute(actionRequest, options = {}) {
    let actionRequestWithDefaults = actionRequest;

    // Default for `timeout` applied here if not defined on request
    if (!actionRequestWithDefaults.parameters.timeout) {
      actionRequestWithDefaults = {
        ...actionRequest,
        parameters: {
          ...actionRequest.parameters,
          timeout: _constants.DEFAULT_EXECUTE_ACTION_TIMEOUT
        }
      };
    }
    return this.handleResponseAction('execute', actionRequestWithDefaults, options);
  }
  async scan(actionRequest, options = {}) {
    return this.handleResponseAction('scan', actionRequest, options);
  }
  async upload(actionRequest, options = {}) {
    const fleetFiles = await this.options.endpointService.getFleetToHostFilesClient();
    const fileStream = actionRequest.file;
    const {
      file: _,
      parameters: userParams,
      ...actionPayload
    } = actionRequest;
    const uploadParameters = {
      ...userParams,
      file_id: '',
      file_name: '',
      file_sha256: '',
      file_size: 0
    };
    const createdFile = await fleetFiles.create(fileStream, actionPayload.endpoint_ids);
    uploadParameters.file_id = createdFile.id;
    uploadParameters.file_name = createdFile.name;
    uploadParameters.file_sha256 = createdFile.sha256;
    uploadParameters.file_size = createdFile.size;
    const createFileActionOptions = {
      ...actionPayload,
      parameters: uploadParameters,
      command: 'upload'
    };
    try {
      const createdAction = await this.handleResponseAction('upload', createFileActionOptions, options);

      // Update the file meta to include the action id, and if any errors (unlikely),
      // then just log them and still allow api to return success since the action has
      // already been created and potentially dispatched to Endpoint. Action ID is not
      // needed by the Endpoint or fleet-server's API, so no need to fail here
      try {
        await fleetFiles.update(uploadParameters.file_id, {
          actionId: createdAction.id
        });
      } catch (e) {
        this.log.warn(`Attempt to update File meta with Action ID failed: ${e.message}`, e);
      }
      return createdAction;
    } catch (err) {
      if (uploadParameters.file_id) {
        // Try to delete the created file since creating the action threw an error
        try {
          await fleetFiles.delete(uploadParameters.file_id);
        } catch (e) {
          this.log.error(`Attempt to clean up file id [${uploadParameters.file_id}] (after action creation was unsuccessful) failed; ${e.message}`, e);
        }
      }
      throw err;
    }
  }
  async getFileDownload(actionId, fileId) {
    await this.ensureValidActionId(actionId);
    const fleetFiles = await this.options.endpointService.getFleetFromHostFilesClient();
    const file = await fleetFiles.get(fileId);
    if (file.actionId !== actionId) {
      throw new _custom_http_request_error.CustomHttpRequestError(`Invalid file id [${fileId}] for action [${actionId}]`, 400);
    }
    return fleetFiles.download(fileId);
  }
  async getFileInfo(actionId, fileId) {
    await this.ensureValidActionId(actionId);
    const fleetFiles = await this.options.endpointService.getFleetFromHostFilesClient();
    const {
      name,
      id,
      mimeType,
      size,
      status,
      created,
      agents,
      actionId: fileActionId
    } = await fleetFiles.get(fileId);
    if (fileActionId !== actionId) {
      throw new _errors.ResponseActionsClientError(`Invalid file ID. File [${fileId}] not associated with action ID [${actionId}]`);
    }
    return {
      name,
      id,
      mimeType,
      size,
      status,
      created,
      actionId,
      agentId: agents[0],
      agentType: this.agentType
    };
  }
}
exports.EndpointActionsClient = EndpointActionsClient;