"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveLogView = void 0;
var _constants = require("../constants");
var _defaults = require("./defaults");
var _errors = require("./errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const resolveLogView = (logViewId, logViewAttributes, dataViewsService, logSourcesService, config) => {
  if (logViewAttributes.logIndices.type === 'index_name') {
    return resolveLegacyReference(logViewId, logViewAttributes, dataViewsService, config);
  } else if (logViewAttributes.logIndices.type === 'data_view') {
    return resolveDataViewReference(logViewAttributes, dataViewsService);
  } else {
    return resolveKibanaAdvancedSettingReference(logViewId, logViewAttributes, dataViewsService, logSourcesService);
  }
};
exports.resolveLogView = resolveLogView;
const resolveLegacyReference = async (logViewId, logViewAttributes, dataViewsService, config) => {
  var _config$messageFields;
  if (logViewAttributes.logIndices.type !== 'index_name') {
    throw new Error('This function can only resolve legacy references');
  }
  const indices = logViewAttributes.logIndices.indexName;
  const dataViewId = `log-view-${logViewId}`;

  // If we didn't remove the item from the cache here the subsequent call to
  // create would not have any effect
  dataViewsService.clearInstanceCache(dataViewId);
  const dataViewReference = await dataViewsService.create({
    id: dataViewId,
    name: logViewAttributes.name,
    title: indices,
    timeFieldName: _constants.TIMESTAMP_FIELD,
    allowNoIndex: true
  }, false, false).catch(error => {
    throw new _errors.ResolveLogViewError(`Failed to create Data View reference: ${error}`, error);
  });
  return {
    indices,
    timestampField: _constants.TIMESTAMP_FIELD,
    tiebreakerField: _constants.TIEBREAKER_FIELD,
    messageField: (_config$messageFields = config.messageFields) !== null && _config$messageFields !== void 0 ? _config$messageFields : _defaults.defaultLogViewsStaticConfig.messageFields,
    fields: dataViewReference.fields,
    runtimeMappings: {},
    columns: logViewAttributes.logColumns,
    name: logViewAttributes.name,
    description: logViewAttributes.description,
    dataViewReference
  };
};
const resolveDataViewReference = async (logViewAttributes, dataViewsService) => {
  var _dataView$timeFieldNa;
  if (logViewAttributes.logIndices.type !== 'data_view') {
    throw new Error('This function can only resolve Kibana data view references');
  }
  const {
    dataViewId
  } = logViewAttributes.logIndices;
  const dataView = await dataViewsService.get(dataViewId).catch(error => {
    throw new _errors.ResolveLogViewError(`Failed to fetch data view "${dataViewId}": ${error}`, error);
  });
  return {
    indices: dataView.getIndexPattern(),
    timestampField: (_dataView$timeFieldNa = dataView.timeFieldName) !== null && _dataView$timeFieldNa !== void 0 ? _dataView$timeFieldNa : _constants.TIMESTAMP_FIELD,
    tiebreakerField: _constants.TIEBREAKER_FIELD,
    messageField: ['message'],
    fields: dataView.fields,
    runtimeMappings: resolveRuntimeMappings(dataView),
    columns: logViewAttributes.logColumns,
    name: logViewAttributes.name,
    description: logViewAttributes.description,
    dataViewReference: dataView
  };
};
const resolveKibanaAdvancedSettingReference = async (logViewId, logViewAttributes, dataViewsService, logSourcesService) => {
  if (logViewAttributes.logIndices.type !== 'kibana_advanced_setting') {
    throw new Error('This function can only resolve references to the Log Sources Kibana advanced setting');
  }
  const indices = (await logSourcesService.getLogSources()).map(logSource => logSource.indexPattern).join(',');
  const dataViewId = `log-view-${logViewId}`;

  // If we didn't remove the item from the cache here the subsequent call to
  // create would not have any effect
  dataViewsService.clearInstanceCache(dataViewId);
  const dataViewReference = await dataViewsService.create({
    id: dataViewId,
    name: logViewAttributes.name,
    title: indices,
    timeFieldName: _constants.TIMESTAMP_FIELD,
    allowNoIndex: true
  }, false, false).catch(error => {
    throw new _errors.ResolveLogViewError(`Failed to create Data View reference: ${error}`, error);
  });
  return {
    indices,
    timestampField: _constants.TIMESTAMP_FIELD,
    tiebreakerField: _constants.TIEBREAKER_FIELD,
    messageField: ['message'],
    fields: dataViewReference.fields,
    runtimeMappings: {},
    columns: logViewAttributes.logColumns,
    name: logViewAttributes.name,
    description: logViewAttributes.description,
    dataViewReference
  };
};

// this might take other sources of runtime fields into account in the future
const resolveRuntimeMappings = dataView => {
  return dataView.getRuntimeMappings();
};