"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InterceptUserInteractionService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _assert = _interopRequireDefault(require("assert"));
var _server = require("@kbn/core/server");
var _configSchema = require("@kbn/config-schema");
var _saved_objects = require("../saved_objects");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class InterceptUserInteractionService {
  constructor() {
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "savedObjectRef", _saved_objects.interceptInteractionUserRecordSavedObject);
    // returns an id scoped to the current user
    (0, _defineProperty2.default)(this, "getSavedObjectId", (triggerId, userId) => `${triggerId}:${userId}`);
  }
  setup(core, logger) {
    core.savedObjects.registerType(this.savedObjectRef);
    const router = core.http.createRouter();
    router.get({
      path: _constants.TRIGGER_USER_INTERACTION_METADATA_API_ROUTE,
      validate: {
        params: _configSchema.schema.object({
          triggerId: _configSchema.schema.string()
        })
      },
      security: {
        authz: {
          enabled: false,
          reason: 'route is public and provides information about the next product intercept trigger'
        }
      }
    }, async (ctx, request, response) => {
      var _coreRequestHandlerCt;
      const coreRequestHandlerCtx = await ctx.core;
      const userId = (_coreRequestHandlerCt = coreRequestHandlerCtx.security.authc.getCurrentUser()) === null || _coreRequestHandlerCt === void 0 ? void 0 : _coreRequestHandlerCt.profile_uid;
      if (!userId) {
        return response.forbidden();
      }
      const result = await this.getUserInteractionSavedObject(userId, request.params.triggerId);
      return response.ok({
        body: result === null || result === void 0 ? void 0 : result.attributes.metadata
      });
    });
    router.post({
      path: _constants.TRIGGER_USER_INTERACTION_METADATA_API_ROUTE,
      validate: {
        params: _configSchema.schema.object({
          triggerId: _configSchema.schema.string()
        }),
        body: _configSchema.schema.object({
          lastInteractedInterceptId: _configSchema.schema.number()
        })
      },
      security: {
        authz: {
          enabled: false,
          reason: 'This route delegates authorization to SO client.'
        },
        authc: {
          enabled: true
        }
      }
    }, async (ctx, request, response) => {
      var _coreRequestHandlerCt2;
      const coreRequestHandlerCtx = await ctx.core;
      const userId = (_coreRequestHandlerCt2 = coreRequestHandlerCtx.security.authc.getCurrentUser()) === null || _coreRequestHandlerCt2 === void 0 ? void 0 : _coreRequestHandlerCt2.profile_uid;
      if (!userId) {
        return response.forbidden();
      }
      await this.recordUserInteractionForTrigger(userId, request.params.triggerId, request.body);
      return response.created();
    });
  }
  start(core) {
    this.savedObjectsClient = core.savedObjects.createInternalRepository([this.savedObjectRef.name]);
  }
  async getUserInteractionSavedObject(userId, triggerId) {
    _assert.default.ok(this.savedObjectsClient, 'savedObjectsClient is not initialized');
    try {
      const userInteractionSavedObject = await this.savedObjectsClient.get(this.savedObjectRef.name, this.getSavedObjectId(triggerId, userId));
      return userInteractionSavedObject;
    } catch (e) {
      // If the saved object is not found, return null
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
        return null;
      }
      throw e;
    }
  }
  async recordUserInteractionForTrigger(userId, triggerId, data) {
    var _interactionTriggerSa;
    _assert.default.ok(this.savedObjectsClient, 'savedObjectsClient is not initialized');
    let interactionTriggerSavedObject = await this.getUserInteractionSavedObject(userId, triggerId);
    const docId = this.getSavedObjectId(triggerId, userId);
    if (!interactionTriggerSavedObject) {
      interactionTriggerSavedObject = await this.savedObjectsClient.create(this.savedObjectRef.name, {
        userId,
        triggerId,
        metadata: data
      }, {
        id: docId
      });
    } else {
      interactionTriggerSavedObject = await this.savedObjectsClient.update(this.savedObjectRef.name, docId, {
        ...interactionTriggerSavedObject.attributes,
        metadata: {
          ...interactionTriggerSavedObject.attributes.metadata,
          ...data
        }
      }, {
        version: interactionTriggerSavedObject.version
      });
    }
    return (_interactionTriggerSa = interactionTriggerSavedObject) === null || _interactionTriggerSa === void 0 ? void 0 : _interactionTriggerSa.attributes;
  }
}
exports.InterceptUserInteractionService = InterceptUserInteractionService;