"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.routeHandlerFactory = void 0;
var _create = require("@kbn/ml-data-view-utils/actions/create");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _transform = require("../../../../common/types/transform");
var _error_utils = require("../../utils/error_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const routeHandlerFactory = routeDependencies => async (ctx, req, res) => {
  const {
    getCoreStart,
    getDataViewsStart
  } = routeDependencies;
  const coreStart = await getCoreStart();
  const dataViews = await getDataViewsStart();
  const {
    transformId
  } = req.params;
  const {
    createDataView,
    timeFieldName
  } = req.query;
  const response = {
    dataViewsCreated: [],
    dataViewsErrors: [],
    transformsCreated: [],
    errors: []
  };
  const esClient = (await ctx.core).elasticsearch.client;
  try {
    const resp = await esClient.asCurrentUser.transform.putTransform({
      // @ts-expect-error @elastic/elasticsearch group_by is expected to be optional in TransformPivot
      body: req.body,
      transform_id: transformId
    });
    if (resp.acknowledged) {
      response.transformsCreated.push({
        transform: transformId
      });
    } else {
      response.errors.push({
        id: transformId,
        error: (0, _error_utils.wrapEsError)(resp)
      });
    }
  } catch (e) {
    response.errors.push({
      id: transformId,
      error: (0, _error_utils.wrapEsError)(e)
    });
  }
  if (createDataView) {
    const {
      savedObjects,
      elasticsearch
    } = coreStart;
    const dataViewsService = await dataViews.dataViewsServiceFactory(savedObjects.getScopedClient(req), elasticsearch.client.asScoped(req).asCurrentUser, req);
    const runtimeMappings = req.body.source.runtime_mappings;
    const {
      dataViewsCreated,
      dataViewsErrors
    } = await (0, _create.createDataViewFn)({
      dataViewsService,
      dataViewName: req.body.dest.index,
      // Adding runtime mappings for transforms of type latest only here
      // since only they will want to replicate the source index mapping.
      // Pivot type transforms have index mappings that cannot be
      // inferred from the source index.
      runtimeMappings: (0, _mlIsPopulatedObject.isPopulatedObject)(runtimeMappings) && (0, _transform.isLatestTransform)(req.body) ? runtimeMappings : {},
      timeFieldName,
      errorFallbackId: transformId
    });
    response.dataViewsCreated = dataViewsCreated;
    response.dataViewsErrors = dataViewsErrors;
  }
  return res.ok({
    body: response
  });
};
exports.routeHandlerFactory = routeHandlerFactory;