"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.serializeNestedFormField = exports.format = exports.ALLOWED_FIELDS = void 0;
var _lodash = require("lodash");
var _types = require("../types");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const serializeNestedFormField = fields => {
  const monitorType = fields[_types.ConfigKey.MONITOR_TYPE];
  const monitorFields = {};
  const defaults = _constants.DEFAULT_FIELDS[monitorType];
  Object.keys(defaults).map(key => {
    var _ref, _get;
    /* split key names on dot to handle dot notation fields,
     * which are changed to nested fields by react-hook-form */
    monitorFields[key] = (_ref = (_get = (0, _lodash.get)(fields, key.split('.'))) !== null && _get !== void 0 ? _get : fields[key]) !== null && _ref !== void 0 ? _ref : defaults[key];
  });
  return monitorFields;
};
exports.serializeNestedFormField = serializeNestedFormField;
const ALLOWED_FIELDS = exports.ALLOWED_FIELDS = [_types.ConfigKey.ENABLED, _types.ConfigKey.ALERT_CONFIG];
const formatSchedule = schedule => {
  const isFrequencyInSeconds = schedule.number.endsWith('s');
  return {
    number: isFrequencyInSeconds ? schedule.number.slice(0, -1) : schedule.number,
    unit: isFrequencyInSeconds ? _types.ScheduleUnit.SECONDS : _types.ScheduleUnit.MINUTES
  };
};
const format = (fields, readOnly = false) => {
  const formattedFields = serializeNestedFormField(fields);
  const textAssertion = formattedFields[_types.ConfigKey.TEXT_ASSERTION] ? `
          await page.getByText('${formattedFields[_types.ConfigKey.TEXT_ASSERTION]}').first().waitFor();` : ``;
  formattedFields[_types.ConfigKey.SCHEDULE] = formatSchedule(formattedFields[_types.ConfigKey.SCHEDULE]);
  const formattedMap = {
    [_types.FormMonitorType.SINGLE]: {
      ...formattedFields,
      [_types.ConfigKey.SOURCE_INLINE]: `step('Go to ${formattedFields[_types.ConfigKey.URLS]}', async () => {
          await page.goto('${formattedFields[_types.ConfigKey.URLS]}');${textAssertion}
        });`,
      [_types.ConfigKey.FORM_MONITOR_TYPE]: _types.FormMonitorType.SINGLE
    },
    [_types.FormMonitorType.MULTISTEP]: {
      ...formattedFields,
      [_types.ConfigKey.METADATA]: {
        script_source: {
          is_generated_script: (0, _lodash.get)(fields, 'source.inline.type') === 'recorder' ? true : false,
          file_name: (0, _lodash.get)(fields, 'source.inline.fileName') || ''
        }
      },
      [_types.ConfigKey.FORM_MONITOR_TYPE]: _types.FormMonitorType.MULTISTEP
    },
    [_types.FormMonitorType.HTTP]: {
      ...formattedFields,
      [_types.ConfigKey.METADATA]: {
        is_tls_enabled: fields.isTLSEnabled
      },
      [_types.ConfigKey.FORM_MONITOR_TYPE]: _types.FormMonitorType.HTTP
    },
    [_types.FormMonitorType.TCP]: {
      ...formattedFields,
      [_types.ConfigKey.METADATA]: {
        is_tls_enabled: fields.isTLSEnabled
      },
      [_types.ConfigKey.FORM_MONITOR_TYPE]: _types.FormMonitorType.TCP
    },
    [_types.FormMonitorType.ICMP]: {
      ...formattedFields,
      [_types.ConfigKey.FORM_MONITOR_TYPE]: _types.FormMonitorType.ICMP
    }
  };
  const formFields = formattedMap[fields[_types.ConfigKey.FORM_MONITOR_TYPE]];
  return readOnly ? (0, _lodash.pick)(formFields, ALLOWED_FIELDS) : formFields;
};
exports.format = format;